# $Id: hdfv.tcl,v 1.13 1997/05/31 08:43:30 yotam Exp yotam $
#
#  HDF viewer
#
#  Yotam Medini / February 1996
#
########################################################################

########################################################################
# Debug routines. copied from Linux Journal October 1995.
# Mostly of Stephen Uhler.
########################################################################

set DEBUG 0;
if {$DEBUG} {

 ########################################################################
 proc dputs {args} {
    puts stderr $args;
 }; # #dpts


 ########################################################################
 proc show {current} {
   if {$current > 0} {
     set info [info level $current]
     set proc [lindex $info 0]
     puts stderr "$current: Procedure $proc \
		  {[info args $proc]}"
     set index 0
     foreach arg [info args $proc] {
       puts stderr \
	    "\t$arg = [lindex $info [incr index]]"
     }
   } else {
     puts stderr "Top level"
   }
 }; # show


 ########################################################################
 proc breakpoint {} {
   set max [expr [info level] - 2];
   set current $max;
   show $current;
     while {1} {
     puts -nonewline stderr "#$current: "
     gets stdin line
     while {![info complete $line]} {
       puts -nonewline stderr "? "
       append line \n[gets stin]
     }
     switch -- $line {
       + {if {$current < $max} {
	   show [incr current]
	   }
	 }
       - {if {$current < $max} {
	   show [incr current -1]
	   }
	 }
       C {puts stderr "Resuming execution"; return}
       ? {show $current}
       default {
	 catch { uplevel $current $line } result
	 puts stderr $result
       }
     }
   }
 }; # breakpoint


 ########################################################################
 proc dassert {cond {msg "Assertion Failed"}} {
    if {!$cond} {
       #dpts stderr $msg;
       breakpoint;
    }
 }; # dassert

} else { # ! DEBUG - dummies
 proc dputs {args} {};
 proc dassert {cond {msg ""}} {};
}

########################################################################
# Given basename - find file
proc findFile {base} {
   global  hdfvGlob env;
   set     result "";
   if [file exists $base] {  # absolute or in local directory
      set result $base;
   } else {

      # Add executable directory to the search path
      set slash [string last / $hdfvGlob(arg0)];
      set searchPath $env(PATH);
      #dpts "findFile: srchPath=$searchPath, slash=$slash arg0=$hdfvGlob(arg0)";
      if {$slash >= 0} {
         set dir [string range $hdfvGlob(arg0) 0 [expr $slash-1]];
         set searchPath [format "%s:%s" $dir $searchPath];
         #dpts "findFile: searchPath=$searchPath";
      }

      while {[string compare $searchPath ""] && ![string compare $result ""]} {
         set colon [string first : $searchPath];
         if {$colon >= 0} {
            set dir [string range $searchPath 0 [expr $colon - 1]];
            set searchPath [string range $searchPath [expr $colon + 1] end];
         } else {
            set dir $searchPath;
            set searchPath "";
         }
         set candid [format %s/%s $dir $base];
         if {[file exists $candid]} {
            set result $candid;
         }
      }
   }
   return $result;
}; # findFile


########################################################################
proc setIcon {win iconWin iconName} {
   global hdfvGlob;

   #dpts ">>setIcon win=$win iconWin=$iconWin iconName=$iconName";
   if {![info exists hdfvGlob(iconImage)]} {
      set iconGif [findFile hdfv.gif];
      #dpts "iconGif=$iconGif";
      if {![string compare $iconGif ""]} {
         warn "Icon hdfv.gif file not found";
      } else {
         set hdfvGlob(iconImage) [image create photo  -file $iconGif];
      }
   }

   #dpts "setIcon1 iconWin=$iconWin";

   if {[info exists hdfvGlob(iconImage)]} {                      #dpts setIcon2
      if {![winfo exists $iconWin]} {
         toplevel $iconWin;                                      #dpts setIc3
         label $iconWin.lbl -image $hdfvGlob(iconImage);
         pack $iconWin.lbl;
      }
      wm iconwindow $win $iconWin;
      wm iconname $win $iconName;
      # wm iconname $iconWin iconName;
   } else {                                                     #dpts setIcon8
      wm iconname $win [format "TfmPk %s" $iconName];
   }
   #dpts "<<setIcon";
}; # setIcon


########################################################################
# Fixed window with icon
########################################################################
proc winFixedIcon {w title iconName} {
   # #dpts "winFixedIcon: "
   #dpts "winFixedIcon: w=$w title=$title, iconName=$iconName";
   wm title $w "$title";
   wm resizable $w 0 0;
   set  wIcon  [format "%sicon" $w];
   #dpts "wIcon=$wIcon";
   setIcon $w $wIcon $iconName;
   return $w;
}; # winFixedIcon


########################################################################
# General top-level fixed window with icon
########################################################################
proc topLvlFixedIcon {wn title {iconName ""} args} {
   #dpts "tpLvlFixdIcon: wn=$wn title=$title, icnTitl=$iconName, args=$args";
   eval toplevel $wn $args;
   winFixedIcon $wn $title $iconName;
   return $wn;
}; # topLvlFixedIcon


########################################################################
# destroy toplevel callback
########################################################################
proc destroyCallback {win} {
   #dpts ">>destroyCallback win=$win";
   set iconWin [wm iconwindow $win];
   #dpts "destroyCallback: iconWin=$iconWin";
   safeDestroy $iconWin;
   #dpts "<<destroyCallback";
}; # destroyCallback


########################################################################
# Warning. Wait for User's OK
########################################################################
proc warn {msg} {
   tk_messageBox -icon warning -message $msg \
      -parent . -title "HDFV - Warning" -type ok;
}; # warn


########################################################################
# Integer Id generator
########################################################################
proc nextId {} {
   global  hdfvGlob;
   set id $hdfvGlob(nextId);
   incr hdfvGlob(nextId);
   return $id;
}; # nextId


########################################################################
# minimize variable by value
########################################################################
proc minimize {varName val} {
   upvar $varName var;
   if {$var > $val} {
      set var $val;
   }
}; # maximize


########################################################################
# maximize variable by value
########################################################################
proc maximize {varName val} {
   upvar $varName var;
   if {$var < $val} {
      set var $val;
   }
}; # maximize


########################################################################
# like x ? y : z - but y,z are any string
proc sifthenelse {cond sTrue sFalse} {
   set s $sFalse;
   if {$cond} {
      set s $sTrue;
   }
   return $s;
}; # sifthenelse


########################################################################
# Put Id+Name+Class for vgroup or vdata
########################################################################
proc idNameClass {fn frmWin id fbgd}  {
   global hdfvGlob;
   set frmWin [frame $frmWin -background $fbgd];
   set finc [frame $frmWin.finc];
   pack $finc  -padx 8 -pady 3 -fill both;
   set fLft [frame $finc.fLft];
   set fRgt [frame $finc.fRgt];
   pack $fLft -side left;
   pack $fRgt -side right;
   set v(Id) $id;
   set v(Name) [$fn name $id];
   set v(Class) [$fn class $id];
   foreach i {Id Name Class} {
      set l [label $fLft.l$i -text $i:]; #  -font $hdfvGlob(boldFont)];
      pack $l -side top -anchor w;
      set l [label $fRgt.v$i -text $v($i) -font $hdfvGlob(idnmFont)];
      pack $l -side top -anchor w;
   }
   return $frmWin;
}; # idNameClass


########################################################################
# Scroll all data fields listboxes (called from scroll bar)
########################################################################
proc dataScroll {fn id bigf ndatafield ndatarecord args} {
   #dpts ">>dataScroll: $bigf $ndatafield $ndatarecord args=$args";
   set op [lindex $args 0];
   if {[string compare $op 0]} {
      set s [lindex $args 1];
      if {![string compare $op moveto]} {
         set offset [expr round($s * $ndatarecord)];
      } else {
         # dassert [expr [string compare $op scroll] == 0] "scroll expected";
         set lbx $bigf.lbx0;
         set y0 [lindex [$lbx yview] 0];
         set yi [expr round($y0 * [$lbx size])];
         set currOffset [$lbx get $yi];
         set what [lindex $args 2];
         if {![string compare $what units]} {
            set offset [expr $currOffset + $s];
         } else {
            # dassert [expr [string compare $what pages] == 0] "pages expected";
            set offset [expr $currOffset + 10 * $s];
         }
      }
      maximize offset 0;
      minimize offset [expr $ndatarecord - 10];
      dataShowOffset $fn $id $bigf $ndatafield $ndatarecord $offset;
   }
   #dpts "<<dataScroll";
}; # dataScroll


########################################################################
# Test if offest candidate is cached with its whole page
########################################################################
proc offsetIsCached {currOffset offset} {
   global hdfvGlob;
   set cached [expr ($currOffset <= $offset) && \
              ($offset + 10 <= $currOffset + $hdfvGlob(cacheSize))];
   #dpts "offsetIsCached cur=$currOffset ofs=$offset r=$cached";
   return $cached;
}; # offsetIsCached


########################################################################
# get the maximum of string lengths of given list (passed by name)
########################################################################
proc maxStrLen {lsName} {
   upvar $lsName ls;
   set m 0;
   foreach e $ls {
      set l [string length $e];
      maximize m $l;
   }
   return $m;
}; # maxStrLen


########################################################################
# data of offset
########################################################################
proc dataShowOffset {fn id bigf ndatafield ndatarecord offset} {
   #dpts "dataShowOffset: fn=$fn, id=$id offset=$offset";
   #dpts "... bigf=$bigf ndatafield=$ndatafield ndatarecord=$ndatarecord";
   global hdfvGlob;

   # The 1st lisbox (fi = -1) holds the index unless it is empty.
   set lbx $bigf.lbx0
   set currOffset -$hdfvGlob(cacheSize); # ensure read
   if {[$lbx size]} {
      set currOffset [$lbx get 0];
      set riBound [expr $currOffset+$hdfvGlob(cacheSize)];
      minimize riBound $ndatarecord;
   }
   set hdfvGlob(seek$fn$id) $offset;
   set cached [offsetIsCached $currOffset $offset];
   #dpts "currOffset=$currOffset, cached=$cached";
   if {!$cached} {
      set half    [expr $hdfvGlob(cacheSize)/2];
      set quarter [expr $hdfvGlob(cacheSize)/4];
      set a [expr $offset - $quarter];
      set currOffset 0;
      if {$a >= 0} {
         set d [expr round(floor((($a + 0.) / $half)))];
         #dpts "d=$d";
         set currOffset [expr $d * $half];
      }
      #dpts "a=$a half=$half currOffset=$currOffset";
      # Debug-assertions
      set nextcached [offsetIsCached $currOffset $offset];
      if {!$nextcached} {
         warn "Cache Calculation wrong!"
      }

      set riBound [expr $currOffset+$hdfvGlob(cacheSize)];
      minimize riBound $ndatarecord;
      set riLast [expr $riBound-1];

      for {set fi -1} {$fi < $ndatafield} {incr fi} {
         set ci [expr $fi+1]; # column index
         set lbx $bigf.lbx$ci;
         set lbWidth [$lbx cget -width];
         $lbx delete 0 end;

         if {$fi < 0} { # Trivial counter field
            set nDig [string length $ndatarecord];
            set fmt [format "%%%ds" $nDig];
            set fieldLabel [format "\#=%d" $ndatarecord];
            set data {};
            for {set ri $currOffset} {$ri < $riBound} {incr ri} {
               lappend data [format $fmt $ri];
            }
         } else {
            set fieldDef [$fn datafield $id $fi];
            set fieldType [lindex $fieldDef 2];
            set data [$fn data $id $fi $currOffset $riLast] ;

            # For character - special format
            if {![string compare $fieldType "DFNT_CHAR8"]} {
               set newData {};
               foreach d $data {
                  set nd "";
                  foreach c $d {
                     if {32 < $c && $c < 128} {
                        set nd [format "%s%c" $nd $c];
                     }
                  }
                  set nd [format "%s %3d" $nd $d];
                  lappend newData $nd;
               }
               set data $newData;
            }
            set maxDataWidth [maxStrLen data];
            maximize lbWidth $maxDataWidth;
            $lbx configure -width $lbWidth;
         }
         eval {$lbx insert 0} $data;
      }; # for fi
   }; # !cached

   set lbxSize [expr $riBound - $currOffset];
   set index [expr ($offset - $currOffset)];
   #dpts "index=$index, offset=$offset, lbxSize=$lbxSize";
   for {set fi -1} {$fi < $ndatafield} {incr fi} {
      set ci [expr $fi+1]; # column index
      set lbx $bigf.lbx$ci;
      $lbx yview $index;
   }
}; # dataShowOffset


########################################################################
# listboxes callback to update the scroll bar.
# Non trivial since the listbox may contain only cached portion of data.
########################################################################
proc updScroolbar {bigf sy ndatarecord args} {
   global hdfvGlob DEBUG;
   if {$DEBUG} {
      #dpts "updScroolbar: ";
      #dpts "bigf=$bigf sy=$sy ndatarecord=$ndatarecord args=$args";
      set curr [$sy get]; #dpts "get: $curr";
   }
   set lbx $bigf.lbx0
   set sz [$lbx size];
   set currOffset [$lbx get 0];
   set y0y1 [$lbx yview];
   #dpts "currOffset=$currOffset size=$sz y0y1=$y0y1";
   for {set i 0} {$i < 2} {incr i} {
      set y [lindex $y0y1 $i];
      set yi [expr $y * $sz];
      set b [expr $currOffset + $yi];
      set s($i) [expr ($b + 0.)/$ndatarecord];
      #dpts "i=$i, y=$y, yi=$yi, b=$b, s(i)=$s($i)";
   }
   $sy set $s(0) $s(1);
}; # updScroolbar


########################################################################
# Seek data record
########################################################################
proc seekData {fn id seekFrm bigf ndatafield ndatarecord} {
   global hdfvGlob;
   set oldOffset $hdfvGlob(seek$fn$id);
   $seekFrm.but configure -state normal;

   set hdfvGlob(seekCommit) 0;
   grab set $seekFrm;
   tkwait variable hdfvGlob(seekCommit);
   grab release $seekFrm;
   $seekFrm.but configure -state disabled;
   focus $bigf;

   set scanned [scan $hdfvGlob(seek$fn$id) "%d" newOffset];
   if {$scanned} {
      minimize newOffset [expr $ndatarecord - 10];
      maximize newOffset 0;
      set $hdfvGlob(seek$fn$id) $newOffset;
      dataShowOffset $fn $id $bigf $ndatafield $ndatarecord  $newOffset;
   } else {
      warn "Integer expected in $hdfvGlob(seek$fn$id)";
      set hdfvGlob(seek$fn$id) $oldOffset;
   }
}; # seekData


########################################################################
# vdataShow
########################################################################
proc dataShow {fn id} {
   #dpts "dataShow: fn=$fn, id=$id";
   global hdfvGlob;

   set dataId $hdfvGlob(id$fn);
   set winName [format ".winF%dI%d" $dataId $id];

   if {[winfo exists $winName]} {
      #dpts "raising winName=$winName";
      wm deiconify $winName;
      raise $winName;
   } else {
      #dpts "dataShow: Creating winName=$winName";
      set bn [file tail $fn];
      set bnid [format "%s%s" $bn $id];
      topLvlFixedIcon $winName "HDFV - $bn $id" $bnid;
      lappend hdfvGlob(winlist$dataId) $winName;

      set finc [idNameClass $fn $winName.finc $id SandyBrown];
      pack $finc -side top -padx 3 -pady 3;

      set ndatafield [$fn ndatafield $id];
      set ndatarecord [$fn ndatarecord $id];
      #dpts "dataShow: $ndatafield=ndatafield, ndatarecord=$ndatarecord";

      set bigf [frame $winName.bigf];
      if {$ndatarecord > $hdfvGlob(minSizeSeek)} {
         #dpts "setting seek";
	 set seekFrm [frame $winName.seekFrm];
	 pack $seekFrm -side top -anchor w;
	 set seekBut [button $seekFrm.but -text Seek: -state disabled \
	              -command [list set hdfvGlob(seekCommit) 1]];
         pack $seekBut -side left -padx 10;
	 set seekEnt [entry $seekFrm.ent -textvariable hdfvGlob(seek$fn$id) \
		      -width 8];
         bind $seekEnt <FocusIn> [list \
	      seekData $fn $id $seekFrm $bigf $ndatafield $ndatarecord];
         bind $seekEnt <Return> [list set hdfvGlob(seekCommit) 1];
	 pack $seekEnt -side left;
      }

      pack $bigf -side top -padx 3;
      for {set fi -1} {$fi < $ndatafield} {incr fi} {
         set ci [expr $fi+1]; # column index
         if {$fi < 0} { # Trivial counter field
            set nDig [string length $ndatarecord];
            set fmt [format "%%%ds" $nDig];
            set fieldLabel [format "\#=%d" $ndatarecord];
         } else {
            # Get the field definition and format it for label
            set fieldDef [$fn datafield $id $fi];
            #dpts "fieldDef=$fieldDef"
            set fieldName [lindex $fieldDef 0];
            set fieldType [lindex $fieldDef 2];
            set fieldOrder [lindex $fieldDef 3];
            set typeHead [string range $fieldType 0 4]
            if {[string compare $typeHead DFNT_] == 0} { # trim leading DFNT_
               set fieldType [string range $fieldType 5 end];
            }
            if {$fieldOrder == 1} {
               set fieldOrder "";
	    } else {
               set fieldOrder [format "\[%d\]" $fieldOrder];
            }
            set fieldLabel [format "%s %s%s" \
                            $fieldName $fieldType $fieldOrder];
         }

         set fl [label $bigf.lbl$ci -text $fieldLabel];
	 grid $fl -column $ci -row 0 -padx 2 -pady 5;
         # pack $fl -side top -pady 5;

         # width of listbox maximizes the label+data widths
         set lbWidth [string length $fieldLabel];

         set lh 10;
         if {$ndatarecord < 10} {
            set lh $ndatarecord;
         }
         if {$fi < 0} {
            if {$ndatarecord <= 10} {
               set lbx [listbox $bigf.lbx0];
            } else {
               set lbx [listbox $bigf.lbx0 \
                        -yscrollcommand [list updScroolbar \
                                         $bigf $bigf.dsy $ndatarecord]];
            }
         } else {
            set sx $bigf.sx$ci;
            set lbx [listbox $bigf.lbx$ci -xscrollcommand [list $sx set]]
            scrollbar $sx -orient horizontal -command [list $lbx xview];
	    grid $sx -column $ci -row 2 -sticky we -padx 2;
         }
         $lbx configure -width $lbWidth -height $lh;
	 grid $lbx -column $ci -row 1 -padx 2;
      }

      # common y-scroll for all fields with padding
      if {$ndatarecord > 10}  {

         set dsy [scrollbar $bigf.dsy -orient vertical \
                  -command [list dataScroll $fn $id $bigf \
                      $ndatafield $ndatarecord]];
         set ci [expr $ndatafield+1];
         grid $dsy -column $ci -row 1 -sticky ns -padx 5;
      }
      dataShowOffset $fn $id $bigf $ndatafield $ndatarecord 0;

      set bc [button $winName.close \
              -text Close -command [list destroy $winName]];
      pack $bc -side bottom -pady 10;
   }

}; # dataShow


########################################################################
# subShow - a callback generating vgroup or vdata window
########################################################################
proc subShow {fn win lbx gd} {
   #dpts ">>subShow: $fn $win $lbx $gd";
   global hdfvGlob;

   set currsel [$lbx curselection];
   if {[llength cursel] != 1} {
      warn "subShow: expected unique selection, currsel=$currsel";
   }
   set item [$lbx get $currsel];
   scan $item "%d" id;
   #dpts "subShow: item=$item, id=$id";

   if {[string compare $gd d]} {
      #dpts "subShow: vgroup";
      # See if there's keep and it off
      set keepVal 1;
      set keep $win.fbot.keep;
      if {[winfo exists $keep]} {
         set keepVal $hdfvGlob(keep$win);
      }
      #dpts "keepVal=$keepVal";
      # if keep make window, otherwise use same window
      if {$keepVal} {
         groupWindow $fn $id;
      } else {
         # after, since this command is destroyed!
         lappend hdfvGlob(gList$win) $id;
         #dpts "subShow: gList= $hdfvGlob(gList$win)";
         $win.fbot.back configure -state normal;
         after idle groupWindow $fn $id $win;
      }


   } else {
      #dpts "subShow: vdata";
      dataShow $fn $id;
   }
   #dpts "<<subShow";
}; # subShow


########################################################################
# Scroll id+name+class
########################################################################
proc idNameClassScroll {bigf args} {
   #dpts "idNameClassScroll: $bigf $args";
   foreach inc {Id Name Class} {
      set lbx $bigf.lbx$inc;
      eval $lbx yview $args;
   }
}; # idNameClassScroll


########################################################################
# Given an Id list, make a list box with name/class
########################################################################
proc mkListBox {win fn labTxt listName gd} {
   global hdfvGlob;
   upvar $listName idList
   #dpts "mkListBox: listName=$listName, idList=$idList";
   set f0 "";
   set n [llength $idList];
   if {$n} {
      set bgd [sifthenelse [string compare $gd g] SandyBrown green4];
      set parwin $win.ft;
      set f0 [frame $parwin.$gd -background $bgd];
      # pack $f0 -side left -padx 10 -pady 10 -anchor c -fill x;
      set f [frame $f0.f];
      grid $f  -padx 3 -pady 3 -ipadx 2 -ipady 2;
      set labTxt [format "%s\[%d\]" $labTxt $n];
      label $f.lbl -text $labTxt;
      set nameList {};
      set classList {};
      set maxLen(Id) 0;  set maxLen(Name) 0;  set maxLen(Class) 0;
      foreach id $idList {
         set name [$fn name $id];
         set class [$fn class $id];
         #dpts "mkListBox: id=$id, name=$name, class=$class";
         lappend nameList $name;
         lappend classList $class;
         # updtae max-lengths
         maximize maxLen(Id) [string length $id];
         maximize maxLen(Name) [string length $name];
         maximize maxLen(Class) [string length $class];
      }

      foreach fooFoo {{id Id} {name Name} {class Class}} {
         set foo [lindex $fooFoo 0];
         set Foo [lindex $fooFoo 1];
         label $f.$foo -text $Foo; # -font $hdfvGlob(boldFont);
         set w [expr $maxLen($Foo)+1];
         set lbx $f.lbx$Foo;
         if {$n > 8} {
            listbox $lbx -width $w -height 8 -yscrollcommand [list $f.fsy set] ;
         } else {
            listbox $lbx -width $w -height $n;
         }
         set dollarList [format "\$%sList" $foo];
         set vdollar [eval concat $dollarList];
         #dpts "dollarList=$dollarList, vdollar=$vdollar";
         eval {$lbx insert 0} $vdollar;
         if {![string compare $foo "id"]} {
            $lbx configure -cursor hand1;
         } else {
            $lbx configure -cursor sb_left_arrow;
         }
      }
      set dashNull "";
      set fsy "";
      if {$n > 8} {
         set fsy [scrollbar $f.fsy -orient vertical \
                  -command [list idNameClassScroll $f]];
	 set dashNull "-";
      }

      eval grid  $f.lbl    -           -            $dashNull;
           grid  $f.id     $f.name     $f.class;
      eval grid  $f.lbxId  $f.lbxName  $f.lbxClass  $fsy;
      if {[string compare $fsy ""]} {
         grid $fsy -sticky ns;
      }

      # recurse to sub vgroup or vdata
      bind $f.lbxId <ButtonRelease-1> [list subShow $fn $win $f.lbxId $gd];
   }
   return $f0;
}; # mkListBox


########################################################################
# Back button callback
########################################################################
proc gback {fn win} {
   global hdfvGlob;
   set gList $hdfvGlob(gList$win);
   set ng [llength $gList];
   set last [expr $ng-1];
   #dpts ">>gback win=$win, ng=$ng, last=$last, gList=$gList";
   set gList [lreplace $gList $last $last]; # pop last element
   set id [lindex $gList [expr $last-1]]; # new last element
   if {$last == 1} {
      $win.fbot.back configure -state disabled;
   }
   set hdfvGlob(gList$win) $gList
   groupWindow $fn $id $win;
   #dpts "<<gback, gList=$hdfvGlob(gList$win)";
}; # gback


########################################################################
# Keep button callback
########################################################################
proc keep {win} {
   global hdfvGlob;
   #dpts ">>keep win=$win";
   set keepVal $hdfvGlob(keep$win);
   if {$keepVal} {
      $win.fbot.back configure -state disabled;
   } else {
      set ng [llength $hdfvGlob(gList$win)];
      #dpts "keep: ng=$ng";
      if {$ng > 1} {
         $win.fbot.back configure -state normal;
      }
   }
   #dpts "<<keep: keepVal=$keepVal";
}; # keep


########################################################################
# Open vgroup or main data window
########################################################################
# The machinery of moving groups within window requires memorizing
# back-list id's. Since a vgroup may have more then one parent
# we should allow multi-windows of the same group.
#
proc groupWindow {fn gid {useWin ""}} {
   #dpts ">>groupWindow fn=$fn, gid=$gid";
   global hdfvGlob;

   set dataId $hdfvGlob(id$fn);
   set titleTail [sifthenelse [expr $gid == -1]  "/Top"  "<$gid>"];
   set bn [file tail $fn];
   set winTitle "HDFV - $bn $titleTail";   #dpts "winTitle=$winTitle";
   set found 0;
   foreach winName $hdfvGlob(winlist$dataId) {
      #dpts "winName=$winName";
      if {[winfo exists $winName]} {
         set wint [wm title $winName];  #dpts "wint=$wint";
         if {![string compare $wint $winTitle]} { # can match some
            #dpts "raising winName=$winName";
            wm deiconify $winName;
            raise $winName;
            set found 1;
         }
      }
      # else ... may consider tidying thining out the list
   }
   #dpts "groupWindow: found=$found, useWin=$useWin";

   if {(!$found) || [string compare $useWin ""]} {
      set winName $useWin;
      if {![string compare $winName ""]} {
         set winName [format ".winF%dI%d" $dataId [nextId]];
         #dpts "Creating winName=$winName";
         set bnid [format "%s%s" $bn $gid];
         topLvlFixedIcon $winName $winTitle $bnid;
         lappend hdfvGlob(winlist$dataId) $winName;
         set hdfvGlob(gList$winName) $gid;
         #dpts "groupWindow:, gList=$hdfvGlob(gList$winName)";


         set fbot [frame $winName.fbot];
         pack $fbot -side bottom -pady 10 -fill x;
         set close [button $fbot.close \
                 -text Close -command [list destroy $winName]];
         if {$gid != -1} {
            set back [button $fbot.back -text Back -width 5 \
               -state disabled -command [list gback $fn $winName]];
            set keepVar hdfvGlob(keep$winName);
            set keep [checkbutton $fbot.keep -text Keep -width 5 \
                      -variable $keepVar -command [list keep $winName]];
            grid $back $keep $close -padx 10
         } else {
            pack $close;
         }

      } else {
         #dpts "Use win $useWin";
         # update; # we are detroying commands we are called from!
         wm title $winName $winTitle;
         destroy $winName.finc; # Id Name Class
         destroy $winName.ft;   # tables
      }
      if {$gid == -1} {
         set title(g) "Lone Vgroups";
         set title(d) "Lone Vdata";
         pack [label $winName.tit -text $fn] -side top -padx 3 -pady 5;
      } else {
         set title(g) Vgroups;
         set title(d) Vdata;
         set finc [idNameClass $fn $winName.finc $gid green4];
         #dpts "groupWindow: finc=$finc";
         pack $finc -side top -padx 3 -pady 3;
      }

      set ft [frame $winName.ft];
      pack $ft -side top; #  -fill x;

      set listCmd(g) listsubgroup;
      set listCmd(d) listsubdata;
      foreach gd {g d} {
	 set gdl [$fn $listCmd($gd) $gid];
	 set fmk [mkListBox $winName $fn $title($gd) gdl $gd];
	 if {[string compare $fmk ""]} {
	    pack $fmk -padx 5 -pady 5;
	 }
      }

   }
   #dpts "<< groupWindow";
}; # groupWindow


########################################################################
# Select File and load
########################################################################
proc selectLoadData {} {
   global  hdfvGlob;
   set fn [tk_getOpenFile];
   loadData $fn;
   return $fn;
}; # selectLoadData


########################################################################
# Load HDF File
########################################################################
proc loadData {fn} {
   global hdfvGlob;
   #dpts "loadData $fn;";
   safeDestroy .tlsid;  # not to confuse search operation
   if {[string compare $fn ""]} {
      set dataList [$hdfvGlob(filelistbox) get 0 end];
      set matchIndex [lsearch -exact $dataList $fn];
      if {$matchIndex >= 0} {
         warn "Data file:\n  $fn\nalready loaded.";
      } else {
         set lfn [hdf $fn];
         if {[string compare $lfn $fn]} {
            warn "Failed to load $fn";
         } else {
            $hdfvGlob(filelistbox) insert end $lfn;
            $hdfvGlob(filelistbox) selection clear 0 end;
            set last [$hdfvGlob(filelistbox) size];
            set last [expr $last-1];
            $hdfvGlob(filelistbox) selection set $last;

            # assign integer id to the data
            set id [nextId];
            set hdfvGlob(id$fn) $id;
            set hdfvGlob(winlist$id) {};

            # make sure button are sensitive
            $hdfvGlob(BringTop) configure -state normal;
            $hdfvGlob(Unload) configure -state normal;
            $hdfvGlob(fileMenu) entryconfigure \
                $hdfvGlob(reopIndex) -state normal;
            $hdfvGlob(searchMenuButton) configure -state normal;
            bringTop;
         }
      }
   }
}; # loadData


########################################################################
# Return the index of the selected data file
########################################################################
proc fileSelectedIndex {} {
   global hdfvGlob;
   set currsel [$hdfvGlob(filelistbox) curselection];
   set i 0; # default first
   if {[llength $currsel]} {
      set i [lindex $currsel 0];
   }
   #dpts "fileSelectedIndex: i=$i";
   return $i;
}; # fileSelectedIndex;


########################################################################
# Return the name of the selected data file
########################################################################
proc fileSelected {} {
   global hdfvGlob;
   set i [fileSelectedIndex];
   #dpts "fileSelected: i=$i";
   set file [$hdfvGlob(filelistbox) get $i];
   #dpts "fileSelected: file=$file";
   return $file;
}; # fileSelected


########################################################################
# Bring up the top window of the selected data
########################################################################
proc bringTop {} {
   #dpts BringTop;
   safeDestroy .tlsid; # do not confuse search
   set fn [fileSelected];
   #dpts "BringTop: fn=$fn";
   groupWindow $fn -1;
}; # bringTop


########################################################################
# if exist then destroy
########################################################################
proc safeDestroy {w} {
   if {[winfo exists $w]} {
      #dpts "destroy $w";
      destroy $w;
   }
}; # safeDestroy


########################################################################
# Close all windows of selected data.
# Close the data and remove from displayed list
########################################################################
proc unloadData {} {
   global hdfvGlob;
   #dpts unloadData;
   safeDestroy .tlsid;
   set iData [fileSelectedIndex];
   set nmData [fileSelected];
   #dpts "unloadData: iData=$iData, nmData=$nmData";

   # Go thru all data windows and delete them.
   set dataId $hdfvGlob(id$nmData);
   foreach w $hdfvGlob(winlist$dataId) {
      #dpts "unloadData: w=$w";
      safeDestroy $w;
   }

   $nmData close; # hdf-file closure
   $hdfvGlob(filelistbox) delete $iData;

   if {![$hdfvGlob(filelistbox) size]} {
      $hdfvGlob(BringTop) configure -state disabled;
      $hdfvGlob(Unload) configure -state disabled;
      $hdfvGlob(fileMenu) entryconfigure $hdfvGlob(reopIndex) -state disabled;
      $hdfvGlob(searchMenuButton) configure -state disabled;
   }

   return $nmData;
}; # unloadData


########################################################################
# Unload current selected data (close its windows) and reload it
########################################################################
proc  hdfvReopen {} {
   #dpts "hdfvReopen";

   set nmData [unloadData];
   loadData $nmData;
}; # hdfvReopen


########################################################################
# Unload all data, and destroy main window
########################################################################
proc hdfvExit {} {
   #dpts "hdfvExit";
   global hdfvGlob;
   while {[$hdfvGlob(filelistbox) size]} {
      unloadData;
   }
   destroy .;
}; # hdfvExit


########################################################################
# Try ID set by searchID
########################################################################
proc searchIDTryId {nmData} {
   global hdfvGlob;
   set id -1; # default
   set sid $hdfvGlob(searchID);
   scan $sid "%d" id;
   #dpts "searchIDTryId: sid=$sid, id=$id";
   set idType [$nmData id $id];
   #dpts "searchIDTryId: nmData=$nmData, id=$id, type=$idType";
   if {![string compare $idType "g"]} {
      #dpts "It is Vgroup";
      groupWindow $nmData $id;
      set hdfvGlob(SearchDone) 1;
   } elseif {![string compare $idType "d"]} {
      #dpts "It is Vdata";
      dataShow $nmData $id;
      set hdfvGlob(SearchDone) 1;
   } else {
      warn "Could not find Id=$id in $nmData";
   }
}; # searchIDTryId


########################################################################
# Seach for Vgroup or Vdata Id and show it
########################################################################
proc searchID {} {
   #dpts "searchID";
   global hdfvGlob;
   # set tl [topLvlFixedIcon .tlsid "HDFV - Search Id" Search];
   set tl [toplevel .tlsid];
   wm transient $tl;

   set nmData [fileSelected];
   set msg [label $tl.msg -text "Enter Vgroup-ID or Vdata-ID\n of $nmData"];
   pack   $msg -side top -pady 10;

   set eid [entry $tl.eid -textvariable hdfvGlob(searchID)];
   bind $eid <Return> [list searchIDTryId $nmData];
   pack $eid -side top;

   set fbt [frame $tl.bf];
   pack $fbt -side bottom -pady 12;

   button $fbt.ok -text OK -width 6 -command [list searchIDTryId $nmData];
   pack $fbt.ok -side left -padx 6;

   button $fbt.cancel -text Cancel \
      -command [list set hdfvGlob(SearchDone) 1];
   pack $fbt.cancel -side left -padx 6;


   focus $eid;
   grab $tl;
   tkwait variable hdfvGlob(SearchDone);

   grab release $tl;
   destroy $tl;
}; # searchID


########################################################################
# set widgets state
########################################################################
proc setWidgetsState {hdfvGlobKey args} {
   #dpts "disableWidgets: hdfvGlobKey=$hdfvGlobKey args=$args"
   global hdfvGlob;
   set state disabled;
   if {$hdfvGlob($hdfvGlobKey)} {
      set state normal;
   }
   foreach w $args {
      #dpts "disableWidgets w=w";
      $w configure -state $state;
   }
}; # setWidgetsState


########################################################################
# Valid regular expression - I wish TCL would do it without catch
########################################################################
proc regexpOK {re} {
   # puts "regexpOK re=$re";
   set ok 1;
   if {[catch {regexp -- $re any}]} {
      set ok 0;
   }
   return $ok;
}; # regexpOK


########################################################################
# Validate Search expression
########################################################################
proc searchExpOK {} {
   global hdfvGlob;
   set ok 1;
   foreach nc {name class} {
      if {$ok && $hdfvGlob($nc.searchActive)} {
         if {$hdfvGlob($nc.regExpActive)} {
            set re $hdfvGlob($nc.searchEntry);
            set ok [regexpOK $re];
            if {!$ok} {
               warn "Illegal Regular Expression in $nc: $re";
            }
         }
      }
   }
   return $ok;
}; # searchExpOK


########################################################################
# Do the Vgroup/Vdata search by scanning
########################################################################
proc doSearchScanID {nmData gd} {
   #dpts "searchScanID: nmData=$nmData gd=$gd";
   global hdfvGlob;

   topLvlFixedIcon .scan "HDFV scanning" Scan;

   button .scan.cancel -text Cancel -command {set scanCancel 1};
   pack .scan.cancel -side bottom -pady 6;

   set lbl [label .scan.lbl -bitmap hourglass];
   pack $lbl -side left -padx 20;

   set hdfvGlob(scanmsg) "About to scan $nmData";
   message .scan.msg -aspect 1000 -textvariable hdfvGlob(scanmsg);
   pack .scan.msg -side right -pady 6;

   set searchDone 0;
   set scanCancel 0;

   update;

   set currGrab [grab current .scan];
   grab .scan;

   set funcs {getid sgetid};
   if {![string compare $gd "g"]} {
      set funcs getid;
   } elseif {![string compare $gd "d"]} {
      set funcs {sgetid};
   }

   $hdfvGlob(searchListBox) delete 0 end;
   set hdfvGlob(SelectFoundId) "Search For Id's";
   set nScanned 0;
   set nMacthed 0;
   foreach func $funcs {
      #dpts "func = $func";
      set scanon [expr $scanCancel == 0];
      set id -1;
      while {$scanon} {
         set id [$nmData $func $id];
         #dpts "id=$id";
         # after 1000;
         if {$id == -1} {
            set scanon 0;
         } else {
            incr nScanned;
            set match 1;
            foreach nc {name class} {
               set name [$nmData name $id];
               set class [$nmData class $id];
               if {$match && $hdfvGlob($nc.searchActive)} {
                  set s [$nmData $nc $id];
                  set t $hdfvGlob($nc.searchEntry);
                  if {$hdfvGlob($nc.regExpActive)} {
                     set match [regexp -- $t $s];
                  } else {
                     set match [expr [string compare $s $t] == 0];
                  }
               }
            }
            if {$match} {
               incr nMacthed;
               $hdfvGlob(searchListBox) insert end $id;
            }
            set hdfvGlob(scanmsg) \
               [format "Found %d scanned %d" $nMacthed $nScanned];
            update;
            set $scanon [expr $scanCancel == 0];
         }
      }
   }

   if {$nMacthed} {
      set hdfvGlob(SelectFoundId) "Select from found Id(s)";
   }

   grab release .scan;
   if [string compare $currGrab ""] {
      grab $currGrab;
   }
   destroy .scan;
}; # searchScanID


########################################################################
# Check valid expression and call Vgroup/Vdata search
########################################################################
proc searchScanID {nmData gd} {
   if {[searchExpOK]} {
      doSearchScanID $nmData $gd;
   }
}; # searchScanID


########################################################################
# Show selected Vgroup or Vdata from search listbox
########################################################################
proc selectShowID {nmData lbx} {
   #dpts "selectShowID: $nmData $lbx";

   set currsel [$lbx curselection];
   if {[llength cursel] != 1} {
      warn "selectShowID: expected unique selection, currsel=$currsel";
   }
   set item [$lbx get $currsel];
   scan $item "%d" id;
   #dpts "selectShowID: item=$item, id=$id";

   set gd [$nmData id $id];
   if {![string compare $gd "g"]} {
      groupWindow $nmData $id;
   } elseif {![string compare $gd "d"]} {
      dataShow $nmData $id;
   } else {
      Warn "hdfv: Software Error - id=$id gd=$gd";
   }
   raise .tlsid; # the grabbing window
}; # selectShowID


########################################################################
# Search for Vgroup or Vdata by name/class and show it
########################################################################
proc searchNameClass {gd} {
   #dpts "searchNameClass gd=$gd";
   global  hdfvGlob;

   set variant "Vgroup/Vdata";
   if {![string compare $gd "g"]} {
      set variant Vgroup;
   } elseif {![string compare $gd "d"]} {
      set variant Vdata;
   }
   #dpts searchNameClass1;
   safeDestroy .tlsid;
   #dpts searchNameClass2;
   set tl [topLvlFixedIcon .tlsid "HDFV - Search $variant" Search];
   #dpts searchNameClass3;

   set nmData [fileSelected];
   set msg [label $tl.msg \
      -text "Search $variant of\n$nmData\nby Name/Class"];
   pack   $msg -side top -pady 10;

   set fnc [frame $tl.fnc];
   pack $fnc -side top;

   foreach ncNC {{name Name} {class Class}} {
      set nc [lindex $ncNC 0];
      set NC [lindex $ncNC 1];
      set f [frame $fnc.$nc];
      pack $f -side top;
      if {![info exists hdfvGlob($nc.searchActive)]} {
         set hdfvGlob($nc.searchActive) 1;
         set hdfvGlob($nc.regExpActive) 0;
      }
      set ent [entry $f.ent -width 24 -textvariable hdfvGlob($nc.searchEntry)];
      set cbre [checkbutton $f.cbre -text RegExp \
                -indicatoron 0 -takefocus 0 \
                -variable hdfvGlob($nc.regExpActive)];
      set cbact [checkbutton $f.cbact -text "$NC:" -width 6 \
                 -indicatoron 0 -takefocus 0 \
                 -variable hdfvGlob($nc.searchActive) \
                 -command [list setWidgetsState $nc.searchActive $ent $cbre]];
      pack $cbact $ent $cbre -side left -padx 6;
   }

   set ffoundlbx [frame $tl.flbx];
   pack $ffoundlbx -side top -padx 6 -pady 6
   set hdfvGlob(SelectFoundId) "Search For Id's";
   set msg [label $ffoundlbx.msg -textvariable hdfvGlob(SelectFoundId)];
   pack $msg -side top -pady 3;
   set flbx [frame $ffoundlbx.flbx];
   pack $flbx -side top;

   set lbx [listbox $flbx.lbx -height 5 -cursor hand1 \
            -yscrollcommand [list $flbx.sy set]];
   set hdfvGlob(searchListBox) $lbx;
   pack $lbx -side left -padx 2 -fill y;
   bind $lbx <ButtonRelease-1> [list selectShowID $nmData $lbx]

   set sy [scrollbar $flbx.sy -command [list $lbx yview]];
   pack $sy -side right -padx 2 -fill y;

   set fbt [frame $tl.bf];
   pack $fbt -side bottom -pady 12;

   button $fbt.ok -text (Re)Search -width 10 \
       -command [list searchScanID $nmData $gd];
   pack $fbt.ok -side left -padx 6;

if {0} {
   button $fbt.close -text Close -width 10 \
      -command [list set hdfvGlob(SearchDone) 1];
   pack $fbt.close -side left -padx 6;

   grab $tl;
   tkwait variable hdfvGlob(SearchDone);

   grab release $tl;
   destroy $tl;
}
   button $fbt.close -text Close -width 10 -command [list destroy $tl];
   pack $fbt.close -side left -padx 6;
}; # searchNameClass


########################################################################
# Search menu
########################################################################
proc searchMenu {mb} {
   global hdfvGlob;
   set f [menubutton $mb.search -text Search -state disabled \
         -menu $mb.search.menu];
   pack $f -side left -padx 10 -pady 6;
   set m [menu $mb.search.menu -tearoff 0];
   set hdfvGlob(searchMenuButton) $f;
   set hdfvGlob(searchMenu) $m;
   $m add command -label Id... -command searchID;
   $m add separator;
   $m add command -label Vgroup... -command [list searchNameClass g];
   $m add command -label Vdata... -command [list searchNameClass d];
   $m add command -label Vgroup/Vdata... -command [list searchNameClass gd];
}; # searchMenu


########################################################################
# About text
########################################################################
proc about {} {
   global hdfvGlob;
   set winName .about;
   if {[winfo exists $winName]} {
      #dpts "raising winName=$winName";
      wm deiconify $winName;
      raise $winName;
   } else {
      topLvlFixedIcon $winName "HDFV - About" About;

      button $winName.close -text Close -command [list destroy $winName];
      pack $winName.close -side bottom -pady 6;

      label $winName.info -bitmap info;
      pack $winName.info -side left -padx 6;

      set t [text $winName.text -setgrid true -wrap word \
             -font *-times-medium-r-*-12-* \
	    -width 56 -height 14 -padx 6];
      pack $t -side left  -padx 6 -pady 6;

      $t tag configure boldBig -font *-times-bold-r-*-14-*
      $t tag configure fixed -font fixed

      $t insert end  "              About HDFV $hdfvGlob(version)\n" boldBig;
      $t insert end  \
"\nHDFV is an HDF-file Viewer. HDF is Hierarchical Data Format\
for scientific data. It is defined with C & Fortran API by NCSA.\n";
      $t insert end "HDF is available from: ";
      $t insert end "/anonymous@ftp.ncsa.uiuc.edu:/HDF/\n" fixed;
      $t insert end "The viewer supports the Vgroup/Vdata model.\n";
      $t insert end "\nHDFV uses Tcl/Tk which is available from:\n"
      $t insert end /anonymous@ftp.smli.com:/pub/tcl/" fixed;
      $t insert end  "\n\nHDFV is written by Yotam Medini  ";
      $t insert end  " (yotam@blueneptune.com).\n" fixed;
      $t insert end  "It is available from:\n";
      $t insert end  "/anonymous@ftp.blueneptune.com:/pub/users/yotam/\n" fixed;
      $t insert end  "See also: http://www.blueneptune.com/~yotam/hdfv.html";
      $t configure -state disabled;
      $t configure -state disabled;
   }
}; # about


########################################################################
# Main Window
########################################################################
proc hdfvMain {args} {
   global hdfvGlob DEBUG tk_strictMotif;
   #dpts "hdfvMain: args=$args";

   # set arg0  for sarching files
   set hdfvGlob(arg0) [lindex $args 0];

   set hdfvGlob(nextId) 0;

   #############
   # Preferences
   #############
   set hdfvGlob(MaxField) 40;
   set hdfvGlob(cacheSize)   128;
   set hdfvGlob(minSizeSeek) 512; # above this we will have seek function
   set tk_strictMotif 1;

   # Fonts
   option add Hdfv*font            -Adobe-Helvetica-Bold-R-Normal--*-120-*
   option add Hdfv*Entry*font      -*-*-bold-r-*--*-120-*-*-m-*
   option add Hdfv*Menubutton*font -Adobe-Helvetica-Bold-O-Normal--*-140-*
   option add Hdfv*Listbox*font    fixed
   set hdfvGlob(idnmFont)          -*-Helvetica-medium-*-Normal--*-120-*

   # set hdfvGlob(boldFont) -*-Helvetica-Bold-R-Normal--*-140-*

   # destroy call back.
   bind Toplevel <Destroy> [list destroyCallback %W];

   # set mw [toplevel .main]
   #dpts "Setting .";
   set mw .;
   winFixedIcon . "HDF-Viewer - Main" Main;

   set mb [frame $mw.menubar -highlightthickness 1 -relief raised -bd 4];
   pack $mb -side top -fill x;

   ###########
   # File menu
   ###########
   # -font $hdfvGlob(boldFont)
   set f [menubutton $mb.file -text File -menu $mb.file.menu];
   pack $f -side left -padx 10 -pady 6;
   set m [menu $mb.file.menu -tearoff 0];
   set hdfvGlob(fileMenu) $m;

   $m add command -label Open... -command [list selectLoadData];

   $m add command -label "Reopen Current" -command hdfvReopen -state disabled;
   set hdfvGlob(reopIndex) 1;

   $m add separator;
   $m add command -label Exit -command hdfvExit;

   if {$DEBUG} {
      $m add separator;
      $m add command -label "Reload Tcl-code" -command [list source hdfv.tcl];
   }

   ##########
   searchMenu $mb;

   ##########
   set help [menubutton $mb.help -text Help -menu $mb.help.menu];
   pack $help -side right -padx 10 -pady 6;
   set m [menu $help.menu -tearoff 0];
   $m add command -label About... -command about;


   ##########################
   # List Box of loaded files
   ##########################
   set flf [frame $mw.flf -borderwidth 10];
   pack $flf -side left;
   label $flf.lbl -text "Loaded Files";
   set hdfvGlob(filelistbox) \
       [listbox $flf.list \
            -yscrollcommand [list $flf.sy set] \
            -xscrollcommand [list $flf.sx set] -width 20 -height 5];
   set sy [scrollbar $flf.sy -orient vertical -command [list $flf.list yview]];
   scrollbar $flf.sx -orient horizontal -command [list $flf.list xview];

   grid $flf.lbl -column 0 -row 0 -padx 2 -pady 3;
   grid $flf.list -column 0 -row 1 -padx 2; 
   grid $flf.sy   -column 1 -row 1 -padx 2 -sticky ns; 
   grid $flf.sx   -column 0 -row 2 -padx 2 -sticky we; 

   #################
   # Control buttons
   #################
   set fcb [frame $mw.fcb]
   set hdfvGlob(BringTop) [button $fcb.bt -text BringTop \
                           -state disabled -command bringTop];
   set packpar [list -side top -padx 5 -pady 10 -fill x];
   eval pack $hdfvGlob(BringTop)  $packpar
   set hdfvGlob(Unload) [button $fcb.ul -text Unload \
                           -state disabled -command unloadData];
   eval pack $hdfvGlob(Unload) $packpar;
   pack $fcb -side right;

   #dpts "hdfvMain: calling doCommandLineArgs";
   eval doCommandLineArgs $args;
   return 1;
}; # hdfvMain


########################################################################
# Process command line arguments
########################################################################
proc doCommandLineArgs {args} {
   #dpts "doCommandLineArgs";
   #dpts "doCommandLineArgs: args=$args";
   #global  argc;
   #global  argv;
   global  hdfvGlob;

   set argc [llength $args];
   #dpts "doCommandLineArgs: argc=$argc";
   for {set ai 0} {$ai < $argc}  {incr ai} {
      set v [lindex $args $ai];
      #dpts "arg\[$ai\]=$v";
   }

   if {$argc == 1} {
      usage;
   }
   for {set ai 0} {$ai < $argc}  {incr ai} {
      set opt [lindex $args $ai];
      if {![string compare $opt "-f"]} {
         incr ai;
         set fn [lindex $args $ai];
         loadData $fn;
      }
      if {![string compare $opt "-help"]} {
         usage;
      }
   }
}; # doCommandLineArgs

########################################################################
# Usage
########################################################################
proc usage {} {
   puts "Usage: hdfv \[-f <hdfFile>\] \[-silent\]";
}; # usage


# hdfvMain; # start application
