// $Id: hdfCmd.c,v 1.5 1996/10/28 07:58:45 yotam Exp yotam $
#include <limits.h>
#include <string.h>
#include <strstream.h>

// Sun's  John Ousterhaut's TCL
#include <tcl.h>

// NCSA's hdf
#include <hdf.h>
#include <vg.h>

#include "debug.h"
#include "hdfCmd.h"


////////////////////////////////////////////////////////////////////////
class HDFclass
{
 public:
   HDFclass(): _hdf(FAIL), _filename(0) {};
   HDFclass(int32 id, const char* fn);
   HDFclass(const HDFclass&);
   HDFclass& operator=(const HDFclass&);
   virtual ~HDFclass();

   // convert form HDF num_type (alas not contiguous) to mnemonic
   static const char* typeName(int32 num_type);

   static int   hdfCmd(ClientData, Tcl_Interp* interp, int argc, char** argv);
   static int   sInstanceCmd(ClientData, Tcl_Interp *, int argc, char** argv);
   int          instanceCmd(Tcl_Interp *, int argc, char** argv);
   static void  sDeleteCmd(ClientData);

   int          nSub(ostream& result, int32 id, bool group);
   int          listSub(ostream& result, int32 id,
                        int first, int last, bool grup);
   int          gdNameClass(ostream& result, int32 id, bool getName);
   int          nDataField(ostream& result, int32 id);
   int          nDataRecord(ostream& result, int32 id);
   int          dataField(ostream& result, int32 id, unsigned fi);
   int          data(ostream& result,
                     int32     id,
                     unsigned  fi,
                     int       first,
                     int       last,
                     int       firstComp,
                     int       compLimit
                    );
   int          gdn(ostream& result, int32 id);
   
   int32        hdf() const {return _hdf;};
   const char*  filename() const {return _filename;};
   void         clear();
 private:
   int32        lone(bool group, int32 f, int32* idArr, int32 asize);
   int32        _hdf;
   char*        _filename;
}; // HDFclass


////////////////////////////////////////////////////////////////////////
HDFclass::HDFclass(int32 id, const char* fn) :
   _hdf(id)
{
   _filename = new char[strlen(fn) + 1];
   (void)strcpy(_filename, fn);
} // HDFclass::HDFclass


////////////////////////////////////////////////////////////////////////
HDFclass::~HDFclass()
{
   clear();
} // HDFclass::~HDFclass


////////////////////////////////////////////////////////////////////////
void
HDFclass::clear()
{
   if (_hdf != FAIL)
   {
      Vend(_hdf);
      Hclose(_hdf);
      _hdf = FAIL;
   }

   if (_filename)
   {
      delete [] _filename;
      _filename = 0;
   }
} // HDFclass::clear


////////////////////////////////////////////////////////////////////////
int
HDFclass::sInstanceCmd(ClientData cd, Tcl_Interp* interp, int argc, char** argv)
{
   HDFclass*  hdfClass = (HDFclass *)cd;
   return( hdfClass->instanceCmd(interp, argc, argv) );
} // sInstanceCmd


////////////////////////////////////////////////////////////////////////
static void
freebyDelete(char* p)
{
   delete [] p;
} // freebyDelete



////////////////////////////////////////////////////////////////////////
static ostream&
usage(ostream &os)
{
   os << "Usage: " << endl <<
   " <HDFfilename> close" << endl <<
   " <HDFfilename> nsubgroup <parentId>" << endl <<
   " <HDFfilename> nsubdata  <parentId>" << endl <<
   " <HDFfilename> listsubgroup <parentId> [first [last]]" << endl <<
   " <HDFfilename> listsubdata  <parentId> [first [last]]" << endl <<
   " <HDFfilename> name  <id>" << endl <<
   " <HDFfilename> class <id>" << endl <<
   " <HDFfilename> ndatafield <id>" << endl <<
   " <HDFfilename> ndatarecord <id>" << endl <<
   " <HDFfilename> datafield <id> <fi>" << endl <<
   " <HDFfilename> data <id> <fi> [first [last [firstComp [compLimit]] ]]" << 
   endl <<
   " <HDFfilename> id <id>" << endl <<
   " <HDFfilename> getid <id>" << endl <<
   " <HDFfilename> sgetid <id>" << endl;

   return(os);
} // usage


////////////////////////////////////////////////////////////////////////
int
HDFclass::instanceCmd(Tcl_Interp* interp, int argc, char** argv)
{
   ostrstream  result;
   int         rc = TCL_OK;
   int         id = (argc < 3 ? FAIL : atoi(argv[2]));
   if (argc < 2)
   {
      usage(result) << ends;
   }
   else if (strcmp(argv[1], "close") == 0)
   {
      rc = Tcl_DeleteCommand(interp, argv[0]);
   }
   else if (strcmp(argv[1], "nsubgroup") == 0)
   {
      if (argc != 3)
      {
         rc = TCL_ERROR;
         result << "Usage: <HDFfilename> nsubgroup <parentId>" << ends;
      }
      else
      {
         rc = nSub(result, id, true);
      }
   }
   else if (strcmp(argv[1], "nsubdata") == 0)
   {
      if (argc != 3)
      {
         rc = TCL_ERROR;
         result << "Usage: <HDFfilename> nsubdata <parentId>" << ends;
      }
      else
      {
         rc = nSub(result, id, false);
      }
   }
   else if ((strcmp(argv[1], "listsubgroup") == 0) ||
            (strcmp(argv[1], "listsubdata") == 0))

   {

      int  first = (argc > 3 ? atoi(argv[3]) : 0);
      int  last  = (argc > 4 ? atoi(argv[4]) : -1);

      rc = listSub(result, id, first, last,
                   (strcmp(argv[1], "listsubgroup") == 0));
   }
   else if (strcmp(argv[1], "name") == 0)
   {
      rc = gdNameClass(result, id, true);
   }
   else if (strcmp(argv[1], "class") == 0)
   {
      rc = gdNameClass(result, id, false);
   }
   else if (strcmp(argv[1], "ndatafield") == 0)
   {
      rc = nDataField(result, id);
   }
   else if (strcmp(argv[1], "ndatarecord") == 0)
   {
      rc = nDataRecord(result, id);
   }
   else if ((strcmp(argv[1], "datafield") == 0) ||
            (strcmp(argv[1], "data") == 0))
   {
      int  fi = (argc > 3 ? atoi(argv[3]) : -1);
      if (strcmp(argv[1], "datafield") == 0)
      {
         if (argc != 4)
         {
            rc = TCL_ERROR;
            result << "Usage: <HDFfilename> datafield  <id> <fi>" << ends;
         }
         else
         {
            int  fi = atoi(argv[3]);
            rc = dataField(result, id, fi);
         }
      }
      else // "data"
      {
         if (argc < 4)
         {
            rc = TCL_ERROR;
            result << "Usage: "
          "<HDFfilename> data <id> <fi> [first [last [firstComp [compLimit]]]]"
            << ends;
         }
         else
         {
            int  first = (argc > 4 ? atoi(argv[4]) : 0);
            int  last  = (argc > 5 ? atoi(argv[5]) : -1);
            int  firstComp  = (argc > 6 ? atoi(argv[6]) : 0);
            int  compLimit  = (argc > 7 ? atoi(argv[7]) : -1);

            rc = data(result, id, fi, first, last, firstComp, compLimit);
         }
      }
   }
   else if ((strcmp(argv[1], "id") == 0) ||
            (strcmp(argv[1], "getid") == 0) ||
            (strcmp(argv[1], "sgetid") == 0))
   {
      if (argc != 3)
      {  
         rc = TCL_ERROR;
         result << "Usage: <HDFfilename> " << argv[1] << " <id>" << endl;
      }
      else
      {
         int  id = atoi(argv[2]);
         if ((strcmp(argv[1], "id") == 0))
         {
            rc = gdn(result, id);
         }
         else 
         {
            id = ((strcmp(argv[1], "getid") == 0)
                  ? Vgetid(_hdf, id)
                  : VSgetid(_hdf, id));
            result << id << ends;
         }
      }
   }
   else
   {
      result << "HDFcommand: unknown option: " << argv[1] << ends;
   }

   char*  rs = result.str(); // imply freeze
   Tcl_SetResult(interp, rs, freebyDelete);
   return(rc);
} // instanceCmd


////////////////////////////////////////////////////////////////////////
// Print a given string with separators as a single string.
static ostream &
prStringle(ostream& os, const char* s)
{
   static const char*  sepcharsns = "\"'\\";
   static const char*  sepchars =   "\"'\\ "; // with space

   if (strpbrk(s, sepchars) == 0)
   {
      os << s; // simple
   }
   else
   {
      int   o, i, l = strlen(s);
      char*  safe = new char[2 * l + 3];

      safe[o = 0] = '"';  o++;
      for (i = 0;  i < l;  i++)
      {
         char  c = s[i];
         if (strchr(sepcharsns, c))
         {
            safe[o++] = '\\';
         }
         safe[o++] = c;
      }
      safe[o++] = '"';
      safe[o++] = '\0';
      os << safe;
      delete [] safe;
   }
   return(os);
} // prStringle
      

////////////////////////////////////////////////////////////////////////
int
HDFclass::nSub(ostream& result, int32 id, bool group)
{
   int  rc = TCL_OK;
   int  n;
   if (id == -1)
   {
      n = (group ? Vlone(hdf(), 0, 0) : VSlone(hdf(), 0, 0));
   }
   else
   {
      int32  gid = Vattach(hdf(), id, "r");
      int    npairs = Vntagrefs(gid);
      int    i, status = gid;
      for (n = 0, i = npairs;  (status != FAIL) && i--;)
      {
          int32    tag, ref;
          status = Vgettagref(gid, i, &tag, &ref);
          if ( ( group && (tag == VGDESCTAG)) ||
               (!group && (tag == VSDESCTAG)) )
          {
             n++;
          }
      }
      Vdetach(gid);
   }
   result << n << ends;
   return(rc);
} // HDFclass::nSub


////////////////////////////////////////////////////////////////////////
int32
HDFclass::lone(bool group, int32 f, int32* idArr, int32 asize)
{
   int32  n = (group 
               ? Vlone(f, idArr, asize) 
               : VSlone(f, idArr, asize));
   return(n);
} // HDFclass::lone

////////////////////////////////////////////////////////////////////////
int
HDFclass::listSub(ostream& result, int32 id, int first, int last, bool group)
{
   int  i, n, rc = TCL_OK;
   if (id == -1)
   {
      n = lone(group, hdf(), 0, 0);
      if (0 < last && last < n)
      {
         n = last + 1;
      }
      int32*  ref = new int32[n];
      (void)lone(group, hdf(), ref, n);
      DPRNL("group=" << (int)group << ", n=" << n);
      for (i = first;  i < n;  i++)
      {
         result << ref[i] << ' ';
      }

   }
   else
   {
      int32  gid = Vattach(hdf(), id, "r");
      int    npairs = Vntagrefs(gid);
      int    status = gid;
      bool   done = false;
      if (last < 0)
      {
         last = npairs - 1;
      }
      for (i = n = 0;  (status != FAIL) && (i < npairs) && !done;  i++)
      {
         int32    tag, ref;
         status = Vgettagref(gid, i, &tag, &ref);
         if ( ( group && (tag == VGDESCTAG)) ||
              (!group && (tag == VSDESCTAG)) )
         {
            if (n >= first)
            {
               if (n > last)
               {
                  done = true;
               }
               else
               {
                  result << ref << ' ';
               }
            }
            n++;
         }
      }
      Vdetach(gid);
   }
   result << ends;
   return(rc);
} // HDFclass::listSub


////////////////////////////////////////////////////////////////////////
int
HDFclass::gdNameClass(ostream& result, int32 id, bool getName)
{
   int  rc = TCL_OK, hdf_rc = FAIL;
   char buf[VSNAMELENMAX];  // == VGNAMELENMAX

   // vexistvg + vexistvs do not distinguish between group/data

   if (vexistvg(hdf(), id))
   {
      int32 gid = Vattach(hdf(), id, "r");
      hdf_rc = (getName ? Vgetname(gid, buf) : Vgetclass(gid, buf));
      Vdetach(gid);
   }

   if ((hdf_rc == FAIL) && vexistvs(hdf(), id))
   {
      int32 sid = VSattach(hdf(), id, "r");
      hdf_rc = (getName ? VSgetname(sid, buf) : VSgetclass(sid, buf));
      VSdetach(sid);
   }

   if (hdf_rc == FAIL)
   {
      // error
      buf[0] = '\0';
   }
   prStringle(result, buf) << ends;
   return(rc);
} // HDFclass::gdNameClass


////////////////////////////////////////////////////////////////////////
int
HDFclass::nDataField(ostream& result, int32 id)
{
   int   rc = TCL_OK;
   int32 sid = VSattach(hdf(), id, "r");
   int   nf = VFnfields(sid);
   VSdetach(sid);
   result << nf << ends;
   return(rc);
} // HDFclass::nDataField


////////////////////////////////////////////////////////////////////////
int
HDFclass::nDataRecord(ostream& result, int32 id)
{
   int    rc = TCL_OK;
   int32  nr, sid = VSattach(hdf(), id, "r");
   int32 st = VSQuerycount(sid, &nr);
   if (st == FAIL)
   {
      nr = 0;
   }
   VSdetach(sid);
   result << nr << ends;
   return(rc);
} // HDFclass::nDataRecord


////////////////////////////////////////////////////////////////////////
const char* 
HDFclass::typeName(int32 num_type)
{
   class NN
   {
     public:
       NN(const char* na, int nu): name(na), num(nu) {};
       const char* name;
       int         num;
   };
   static NN  nn[] =
   {
      NN("", -13),           // scrtach
      NN("DFNT_NONE", 0),
      NN("DFNT_VERSION", 1),
      NN("DFNT_FLOAT32", 5),
      NN("DFNT_FLOAT64", 6),
      NN("DFNT_FLOAT128", 7),
      NN("DFNT_INT8", 20),
      NN("DFNT_UINT8", 21),
      NN("DFNT_INT16", 22),
      NN("DFNT_UINT16", 23),
      NN("DFNT_INT32", 24),
      NN("DFNT_UINT32", 25),
      NN("DFNT_INT64", 26),
      NN("DFNT_UINT64", 27),
      NN("DFNT_INT128", 28),
      NN("DFNT_UINT128", 30),
      NN("DFNT_UCHAR8", 3),
      NN("DFNT_CHAR8", 4),
      NN("DFNT_CHAR16", 42),
      NN("DFNT_UCHAR16", 43)
   };

   static const nnSize = (sizeof(nn) / sizeof(nn[0]));
   int          i;
   unsigned     nt = (unsigned)num_type & DFNT_MASK;
   nn[0].num = nt;   // ensure stop in next for-loop
   for (i = nnSize;  nn[--i].num != (int)nt;);
   const char* rs = nn[i].name;
   return(rs);
} // HDFclass::typeName


////////////////////////////////////////////////////////////////////////
int
HDFclass::dataField(ostream& result, int32 id, unsigned fi)
{
   int  rc = TCL_OK;
   int32 sid = VSattach(hdf(), id, "r");
   int   nf = VFnfields(sid);
   if ((int)fi >= nf)
   {
      result << "data has " << nf << " fields (fi=" << fi << ")" << ends;
      rc = TCL_ERROR;
   }
   else
   {
      // <fieldname> <num_type> <typename> order
      const char*      fieldname = VFfieldname(sid, fi);
      int             num_type   = VFfieldtype(sid, fi);
      const char*      tn        = typeName(num_type);
      int             order      = VFfieldorder(sid, fi);
      VSdetach(sid);
      prStringle(result, (strlen(fieldname) ? fieldname : "?")) <<
             ' ' << num_type << ' ' <<
             (strlen(tn) ? tn : "?") << ' ' << order << ends;
   }
   return(rc);
} // HDFclass::dataField


////////////////////////////////////////////////////////////////////////
int
HDFclass::data
(
   ostream&  result,
   int32     id,
   unsigned  fi,
   int       first,
   int       last,
   int       firstComp,
   int       compLimit
)
{
   int    rc = TCL_OK;
   int32  nr, sid = VSattach(hdf(), id, "r");
   int    order, st, nf = VFnfields(sid);
   if ((int)fi >= nf)
   {
      result << "data has " << nf << " fields (fi=" << fi << ")" << ends;
      rc = TCL_ERROR;
   }
   else
   {
      order = VFfieldorder(sid, fi);
      st = VSQuerycount(sid, &nr);
   }
   if ((rc == TCL_OK) && (st != FAIL) && (first < nr) && (last < nr) &&
        ((last == -1) || (first <= last)) &&
        (0 <= firstComp && firstComp < order)
      )
   {
      int             num_type   = VFfieldtype(sid, fi);
      int compUpBnd = (((compLimit == -1) || (firstComp + compLimit >= order))
                       ? order : firstComp + compLimit);
      int nRead = ((last == -1) ? nr - first : last - first + 1);
      int             fs         = VFfieldisize(sid, fi);
      uint8*           buf = new uint8[fs * nRead];
      const char*      fieldname = VFfieldname(sid, fi);
      VSseek(sid, first);
      VSsetfields(sid, fieldname); // just one field
      st = VSread(sid, buf, nRead, FULL_INTERLACE);
      int  ri, ci, flatIndex;
      for (ri = flatIndex = 0;  ri < nRead;  ri++)
      {
         flatIndex += firstComp;
         for (ci = firstComp;  ci < compUpBnd;  ci++, flatIndex++)
         {
            result << (ci == firstComp ? "" : ",");
            // very inefficient to put the swich nested.
            const char* tn =  typeName(num_type);
            switch (num_type)
            {

#define caseHDFtypeCtypeCast(HDFtype,Ctype,cast) \
             case HDFtype: \
               DASSERT((int)fs == (int)(order * sizeof(Ctype))); \
               {  const Ctype* pa = (const Ctype* )buf; \
                  DPRNL0("ri=" << ri << ", ci=" << ci << \
                     ", v=" << cast (pa[flatIndex])); \
                  result << cast (pa[flatIndex]); \
               } \
               break

#define caseHDFtypeCtype(HDFtype,Ctype) caseHDFtypeCtypeCast(HDFtype,Ctype,)

             caseHDFtypeCtype(DFNT_FLOAT32, float);
             caseHDFtypeCtype(DFNT_FLOAT64, double);

             caseHDFtypeCtypeCast(DFNT_INT8, signed char, (int));
             caseHDFtypeCtypeCast(DFNT_UINT8, signed char, (unsigned int));
             caseHDFtypeCtype(DFNT_INT16, short);
             caseHDFtypeCtype(DFNT_UINT16, unsigned short);
             caseHDFtypeCtype(DFNT_INT32, long int);
             caseHDFtypeCtype(DFNT_UINT32, unsigned long);

             caseHDFtypeCtypeCast(DFNT_UCHAR8, unsigned char, (unsigned int));
             caseHDFtypeCtypeCast(DFNT_CHAR8, signed char, (int));

             default:
               result << '?' << tn << '?';                     DASSERT(false);
            }
         }
         result << ' ';
         flatIndex += order - compUpBnd;
      }
      result << ends;                     DASSERT(flatIndex == nRead * order);
      delete [] buf;
   }
   VSdetach(sid);
   return(rc);
} // HDFclass::data


////////////////////////////////////////////////////////////////////////
int
HDFclass::gdn(ostream& result, int32 id)
{
   int32  gid = Vattach(hdf(), id, "r");
   int32  sid = VSattach(hdf(), id, "r");
   DASSERT((gid == FAIL) || (sid == FAIL));
   if (gid != FAIL)
   {
      result << 'g';
      Vdetach(id);
   } 
   else if (sid != FAIL)
   {
      result << 'd';
      Vdetach(id);
   } 
   else
   {
      result << 'n';
   }
   result << ends;
   return(TCL_OK);
} // HDFclass::gdn


////////////////////////////////////////////////////////////////////////
void
HDFclass::sDeleteCmd(ClientData cd)
{
   HDFclass*  hdfClass = (HDFclass *)cd;
   delete hdfClass;
} // sDeleteCmd


////////////////////////////////////////////////////////////////////////
int
HDFclass::hdfCmd(ClientData, Tcl_Interp* interp, int argc, char** argv)
{
   int  rc = TCL_OK;
   ostrstream  result(interp->result, TCL_RESULT_SIZE);
   if (argc != 2)
   {
      rc = TCL_ERROR;
      result << "Usage:  hdfCmd <HDFfilename>" << ends;
   }
   else
   {
      const char* filename = argv[1];
      int32  hdf = Hopen(filename, DFACC_READ, 0);
      bool   ok = (hdf != FAIL) && (Vstart(hdf) != FAIL);
      if (ok)
      {
         HDFclass*  hdfClass = new HDFclass(hdf, filename);
         (void)Tcl_CreateCommand(interp, (char *)hdfClass->filename(),
                                 sInstanceCmd, hdfClass, sDeleteCmd);
      }
      result << (ok ? filename : "") << ends;
   }
   return(rc);
} // hdfCmd



////////////////////////////////////////////////////////////////////////
int
hdf_Init(Tcl_Interp* interp)
{
   Tcl_CreateCommand(interp, "hdf", HDFclass::hdfCmd, 0, 0);
   return(TCL_OK);
} // hdf_Init

