% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/currstatCIR.R
\name{currstatCIR}
\alias{currstatCIR}
\title{Estimate a survival function under current status sampling}
\usage{
currstatCIR(
  time,
  event,
  X,
  SL_control = list(SL.library = c("SL.mean", "SL.glm"), V = 3),
  HAL_control = list(n_bins = c(5), grid_type = c("equal_mass"), V = 3),
  deriv_method = "m-spline",
  eval_region,
  n_eval_pts = 101,
  alpha = 0.05
)
}
\arguments{
\item{time}{\code{n x 1} numeric vector of observed monitoring times. For individuals that were never
monitored, this can be set to any arbitrary value, including \code{NA}, as long as the corresponding
\code{event} variable is \code{NA}.}

\item{event}{\code{n x 1} numeric vector of status indicators of
whether an event was observed prior to the monitoring time. This value must be \code{NA} for
individuals that were never monitored.}

\item{X}{\code{n x p} dataframe of observed covariate values.}

\item{SL_control}{List of \code{SuperLearner} control parameters. This should be a named list; see
\code{SuperLearner} documentation for further information.}

\item{HAL_control}{List of \code{haldensify} control parameters. This should be a named list; see
\code{haldensify} documentation for further information.}

\item{deriv_method}{Method for computing derivative. Options are \code{"m-spline"} (the default,
fit a smoothing spline to the estimated function and differentiate the smooth approximation),
\code{"linear"} (linearly interpolate the estimated function and use the slope of that line), and
\code{"line"} (use the slope of the line connecting the endpoints of the estimated function).}

\item{eval_region}{Region over which to estimate the survival function.}

\item{n_eval_pts}{Number of points in grid on which to evaluate survival function.
The points will be evenly spaced, on the quantile scale, between the endpoints of \code{eval_region}.}

\item{alpha}{The level at which to compute confidence intervals and hypothesis tests. Defaults to 0.05}
}
\value{
Data frame giving results, with columns:
\item{t}{Time at which survival function is estimated}
\item{S_hat_est}{Survival function estimate}
\item{S_hat_cil}{Lower bound of confidence interval}
\item{S_hat_ciu}{Upper bound of confidence interval}
}
\description{
Estimate a survival function under current status sampling
}
\examples{
\dontrun{# This is a small simulation example
set.seed(123)
n <- 300
x <- cbind(2*rbinom(n, size = 1, prob = 0.5)-1,
           2*rbinom(n, size = 1, prob = 0.5)-1)
t <- rweibull(n,
              shape = 0.75,
              scale = exp(0.4*x[,1] - 0.2*x[,2]))
y <- rweibull(n,
              shape = 0.75,
              scale = exp(0.4*x[,1] - 0.2*x[,2]))

# round y to nearest quantile of y, just so there aren't so many unique values
quants <- quantile(y, probs = seq(0, 1, by = 0.05), type = 1)
for (i in 1:length(y)){
  y[i] <- quants[which.min(abs(y[i] - quants))]
}
delta <- as.numeric(t <= y)

dat <- data.frame(y = y, delta = delta, x1 = x[,1], x2 = x[,2])

dat$delta[dat$y > 1.8] <- NA
dat$y[dat$y > 1.8] <- NA
eval_region <- c(0.05, 1.5)
res <- survML::currstatCIR(time = dat$y,
                           event = dat$delta,
                           X = dat[,3:4],
                           SL_control = list(SL.library = c("SL.mean", "SL.glm"),
                                             V = 3),
                           HAL_control = list(n_bins = c(5),
                                              grid_type = c("equal_mass"),
                                              V = 3),
                           eval_region = eval_region)

xvals = res$t
yvals = res$S_hat_est
fn=stepfun(xvals, c(yvals[1], yvals))
plot.function(fn, from=min(xvals), to=max(xvals))}

}
