% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/select_mtd.R
\name{select_mtd}
\alias{select_mtd}
\title{MTD Selection for Multiple Trials}
\usage{
select_mtd(iso_est_mat, n_pts_mat, eliminated_mat, target,
             boundMTD = FALSE, lambda_d = NULL, min_mtd_sample = 1)
}
\arguments{
\item{iso_est_mat}{Numeric matrix of size n_trials x n_doses. Isotonic regression estimates
of toxicity rates from \code{\link{isotonic_regression}}.}

\item{n_pts_mat}{Numeric matrix of size n_trials x n_doses. Number of patients treated at
each dose level for each trial.}

\item{eliminated_mat}{Logical matrix of size n_trials x n_doses. Whether each dose has been
eliminated in each trial.}

\item{target}{Numeric. Target toxicity probability.}

\item{boundMTD}{Logical. If TRUE, impose constraint that selected MTD's isotonic estimate
must be <= lambda_d. Default is FALSE.}

\item{lambda_d}{Numeric. De-escalation boundary. Required if boundMTD = TRUE.}

\item{min_mtd_sample}{Numeric. Minimum number of patients required for a dose to be considered
for MTD selection. Default is 1.}
}
\value{
Data frame with n_trials rows and three columns:
\item{trial}{Integer. Trial ID (1 to n_trials)}
\item{mtd}{Integer. Selected MTD dose number, or NA if no valid dose}
\item{reason}{Character. Reason for trial completion or termination}
}
\description{
Select maximum tolerated dose (MTD) for multiple trials based on isotonic
regression estimates and dose elimination status. This function follows the
BOIN package's MTD selection algorithm.
}
\details{
For each trial, the function performs the following steps:
\enumerate{
\item Check if lowest dose (dose 1) is eliminated first
\item Identify admissible set: doses with patients AND not eliminated
\item Extract isotonic estimates for admissible doses
\item Select dose with estimate closest to target toxicity rate
\item Apply tiebreaker by adding small dose-index increments
\item If boundMTD = TRUE, ensure selected dose satisfies constraint
}

The dose with isotonic estimate closest to the target is selected as MTD.
Ties are broken by small perturbation (1e-10 * dose_index) preferring
lower dose indices when estimates are equally close to target.

If the lowest dose (dose 1) is eliminated, NO MTD is selected regardless
of other doses' status. This follows BOIN standard: "stop the trial if
the lowest dose is eliminated due to toxicity, and no dose should be
selected as the MTD."
}
\examples{
target <- 0.30
n_pts_mat <- matrix(c(3, 6, 9, 3, 6, 9), nrow = 2, byrow = TRUE)
n_tox_mat <- matrix(c(0, 1, 3, 0, 1, 2), nrow = 2, byrow = TRUE)
eliminated_mat <- matrix(FALSE, nrow = 2, ncol = 3)

iso_est_mat <- isotonic_regression(n_pts_mat, n_tox_mat)
mtd_results <- select_mtd(iso_est_mat, n_pts_mat, eliminated_mat, target)
print(mtd_results)

}
