# ============================================================================
# 1. Utility Functions
# ============================================================================

#' @title Determine Number of FPCs Automatically
#' @description Determines the number of functional principal components (FPCs)
#' required to meet a target Proportion of Variance Explained (PVE).
#'
#' @param Y_mat A numeric matrix (N x M) where N is subjects, M is time points.
#' @param argvals A numeric vector of length M listing the observation points.
#' @param target_pve The target cumulative PVE to reach (default: 0.95).
#' @param max_npc The maximum number of components to retain (default: 6L).
#'
#' @return A list containing:
#' \item{npc}{The selected number of components.}
#' \item{pooled_fpca}{The `fpca.sc` object from the `refund` package.}
#' \item{scree}{A tibble with PVE and cumulative PVE for each component.}
#'
#' @export
#' @importFrom refund fpca.sc
#' @importFrom tibble tibble
#'
#' @examples
#' \donttest{
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   npc_info <- determine_npc(sim$Y_mat, sim$argvals)
#'   print(npc_info$npc)
#' }
determine_npc <- function(Y_mat, argvals, target_pve = 0.95, max_npc = 6L) {
  if (!is.matrix(Y_mat)) {
    Y_mat <- as.matrix(Y_mat)
  }
  fpca_full <- refund::fpca.sc(Y = Y_mat, argvals = argvals, pve = target_pve)
  npc <- min(fpca_full$npc, max_npc)
  total_var <- sum(fpca_full$evalues)
  pve_vec <- fpca_full$evalues / total_var
  scree_df <- tibble::tibble(
    npc = seq_along(fpca_full$evalues),
    pve = pve_vec,
    cum_pve = cumsum(pve_vec)
  )
  if (ncol(fpca_full$efunctions) > npc) {
    fpca_full$efunctions <- fpca_full$efunctions[, seq_len(npc), drop = FALSE]
  }
  if (ncol(fpca_full$scores) > npc) {
    fpca_full$scores <- fpca_full$scores[, seq_len(npc), drop = FALSE]
  }
  if (length(fpca_full$evalues) > npc) {
    fpca_full$evalues <- fpca_full$evalues[seq_len(npc)]
  }
  fpca_full$npc <- npc
  list(npc = npc, pooled_fpca = fpca_full, scree = scree_df)
}

#' @title Build Scree Plot for Configural Invariance
#' @description Creates a ggplot2 scree plot comparing the PVE by FPCs for
#' two groups.
#'
#' @param fpca_A An `fpca.sc` object for Group A.
#' @param fpca_B An `fpca.sc` object for Group B.
#' @param groups A character vector of the two group names.
#'
#' @return A ggplot object.
#'
#' @export
#' @importFrom tibble tibble
#' @importFrom ggplot2 ggplot aes geom_line geom_point labs
#' @importFrom ggplot2 scale_x_continuous scale_color_manual theme_bw
#' @importFrom dplyr bind_rows
#' @importFrom stats setNames
#'
#' @examples
#' \donttest{
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   g_fac <- factor(sim$group_vec)
#'   idx_A <- which(g_fac == levels(g_fac)[1])
#'   # --- FIX: Added Y= and argvals= ---
#'   fpca_A <- refund::fpca.sc(Y = sim$Y_mat[idx_A, ], argvals = sim$argvals, npc = 3)
#'   fpca_B <- refund::fpca.sc(Y = sim$Y_mat[-idx_A, ], argvals = sim$argvals, npc = 3)
#'   build_scree_plot(fpca_A, fpca_B, levels(g_fac))
#' }
build_scree_plot <- function(fpca_A, fpca_B, groups) {
  npc <- pve <- group <- NULL
  pve_A <- tibble::tibble(
    npc = seq_along(fpca_A$evalues),
    pve = fpca_A$evalues / sum(fpca_A$evalues),
    group = groups[1]
  )
  pve_B <- tibble::tibble(
    npc = seq_along(fpca_B$evalues),
    pve = fpca_B$evalues / sum(fpca_B$evalues),
    group = groups[2]
  )
  group_colors <- stats::setNames(c("#E69F00", "#0072B2"), groups)
  dplyr::bind_rows(pve_A, pve_B) |>
    ggplot2::ggplot(aes(x = .data$npc, y = .data$pve, group = .data$group, color = .data$group)) +
    ggplot2::geom_line(linewidth = 1) +
    ggplot2::geom_point(size = 3) +
    ggplot2::labs(
      title = "Scree Plot for Configural Invariance Diagnosis",
      x = "Number of FPCs",
      y = "Proportion of Variance Explained (PVE)"
    ) +
    ggplot2::scale_x_continuous(breaks = seq_len(max(pve_A$npc, pve_B$npc))) +
    ggplot2::scale_color_manual(values = group_colors) +
    ggplot2::theme_bw(base_size = 14)
}

# ============================================================================
# 2. Core Test Functions
# ============================================================================

#' @title Test for Configural Invariance
#' @description Permutation test to check if the cumulative PVE is equivalent
#' across groups.
#'
#' @inheritParams run_fmi
#' @return A list with `passed` (boolean), `p_value`, `statistic`, and `plot`.
#' @export
#' @importFrom refund fpca.sc
#' @importFrom utils txtProgressBar setTxtProgressBar
#'
#' @examples
#' \donttest{
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   res <- test_configural(sim$Y_mat, sim$group_vec, sim$argvals,
#'                          npc = 3, n_perms = 9, alpha = 0.05, progress = FALSE)
#'   print(res$p_value)
#' }
test_configural <- function(Y_mat, group_vec, argvals, npc, n_perms, alpha, progress) {
  if (!is.matrix(Y_mat)) {
    Y_mat <- as.matrix(Y_mat)
  }
  group_fac <- factor(group_vec)
  groups <- levels(group_fac)
  idx_A <- which(group_fac == groups[1])
  fpca_A <- refund::fpca.sc(Y = Y_mat[idx_A, ], argvals = argvals, npc = npc)
  fpca_B <- refund::fpca.sc(Y = Y_mat[-idx_A, ], argvals = argvals, npc = npc)
  calc_cum_pve <- function(fpca_obj) cumsum(fpca_obj$evalues) / sum(fpca_obj$evalues)
  observed_stat <- sum((calc_cum_pve(fpca_A) - calc_cum_pve(fpca_B))^2)
  perm_stats <- numeric(n_perms)
  N <- length(group_fac)
  N_A <- length(idx_A)
  if (progress) pb <- utils::txtProgressBar(min = 0, max = n_perms, style = 3)
  for (i in seq_len(n_perms)) {
    perm_ids_A <- sample.int(N, size = N_A, replace = FALSE)
    perm_ids_B <- setdiff(seq_len(N), perm_ids_A)
    fpca_perm_A <- refund::fpca.sc(Y = as.matrix(Y_mat[perm_ids_A, ]), argvals = argvals, npc = npc)
    fpca_perm_B <- refund::fpca.sc(Y = as.matrix(Y_mat[perm_ids_B, ]), argvals = argvals, npc = npc)
    perm_stats[i] <- sum((calc_cum_pve(fpca_perm_A) - calc_cum_pve(fpca_perm_B))^2)
    if (progress) utils::setTxtProgressBar(pb, i)
  }
  if (progress) close(pb)
  p_value <- (sum(perm_stats >= observed_stat) + 1) / (n_perms + 1)
  list(
    passed = p_value >= alpha,
    p_value = p_value,
    statistic = observed_stat,
    plot = build_scree_plot(fpca_A, fpca_B, groups)
  )
}

#' @title Test for Metric Invariance
#' @description Permutation test to check if the eigenfunctions are equivalent
#' across groups.
#'
#' @inheritParams test_configural
#' @return A list with `passed` (boolean), `p_value`, and `statistic`.
#' @export
#' @importFrom refund fpca.sc
#'
#' @examples
#' \donttest{
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   res <- test_metric(sim$Y_mat, sim$group_vec, sim$argvals,
#'                      npc = 3, n_perms = 9, alpha = 0.05, progress = FALSE)
#'   print(res$p_value)
#' }
test_metric <- function(Y_mat, group_vec, argvals, npc, n_perms, alpha, progress) {
  if (!is.matrix(Y_mat)) {
    Y_mat <- as.matrix(Y_mat)
  }
  group_fac <- factor(group_vec)
  idx_A <- which(group_fac == levels(group_fac)[1])
  fpca_A <- refund::fpca.sc(Y = Y_mat[idx_A, ], argvals = argvals, npc = npc)
  fpca_B <- refund::fpca.sc(Y = Y_mat[-idx_A, ], argvals = argvals, npc = npc)
  calc_stat <- function(f_A, f_B, k, delta) {
    sign_corr <- sign(sum(f_A$efunctions[, k] * f_B$efunctions[, k]))
    diff_vec <- f_A$efunctions[, k] - sign_corr * f_B$efunctions[, k]
    sum(diff_vec^2) * delta
  }
  delta <- if (length(argvals) > 1) mean(diff(argvals)) else 1
  observed_stat <- sum(sapply(seq_len(npc), function(k) calc_stat(fpca_A, fpca_B, k, delta)))
  perm_stats <- numeric(n_perms)
  N <- length(group_fac)
  N_A <- length(idx_A)
  if (progress) pb <- utils::txtProgressBar(min = 0, max = n_perms, style = 3)
  for (i in seq_len(n_perms)) {
    perm_ids_A <- sample.int(N, size = N_A, replace = FALSE)
    perm_ids_B <- setdiff(seq_len(N), perm_ids_A)
    fpca_perm_A <- refund::fpca.sc(Y = as.matrix(Y_mat[perm_ids_A, ]), argvals = argvals, npc = npc)
    fpca_perm_B <- refund::fpca.sc(Y = as.matrix(Y_mat[perm_ids_B, ]), argvals = argvals, npc = npc)
    perm_stats[i] <- sum(sapply(seq_len(npc), function(k) calc_stat(fpca_perm_A, fpca_perm_B, k, delta)))
    if (progress) utils::setTxtProgressBar(pb, i)
  }
  if (progress) close(pb)
  p_value <- (sum(perm_stats >= observed_stat) + 1) / (n_perms + 1)
  list(
    passed = p_value >= alpha,
    p_value = p_value,
    statistic = observed_stat
  )
}

#' @title Test for Scalar Invariance
#' @description Permutation test to check if the mean functions (intercepts)
#' are equivalent across groups, after accounting for latent factors.
#'
#' @param Y_mat Numeric matrix (N x M).
#' @param group_vec A vector of length N specifying group membership.
#' @param argvals Numeric vector of length M.
#' @param pooled_fpca The `fpca.sc` object from the `determine_npc` function.
#' @param n_perms Integer, number of permutations.
#' @param alpha Significance level (default: 0.05).
#' @param progress Boolean, show progress bar? (default: interactive()).
#'
#' @return A list with `passed` (boolean), `p_value`, and `statistic`.
#' @export
#'
#' @examples
#' \donttest{
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   npc_info <- determine_npc(sim$Y_mat, sim$argvals, max_npc = 3)
#'   res <- test_scalar(sim$Y_mat, sim$group_vec, sim$argvals,
#'                      pooled_fpca = npc_info$pooled_fpca,
#'                      n_perms = 9, alpha = 0.05, progress = FALSE)
#'   print(res$p_value)
#' }
test_scalar <- function(Y_mat, group_vec, argvals, pooled_fpca, n_perms, alpha, progress) {
  if (!is.matrix(Y_mat)) {
    Y_mat <- as.matrix(Y_mat)
  }
  group_fac <- factor(group_vec)
  groups <- levels(group_fac)
  reconstructed_latent <- pooled_fpca$scores %*% t(pooled_fpca$efunctions)
  residuals_mat <- Y_mat - reconstructed_latent
  mu_resid_A <- colMeans(residuals_mat[group_vec == groups[1], ])
  mu_resid_B <- colMeans(residuals_mat[group_vec == groups[2], ])
  delta <- if (length(argvals) > 1) mean(diff(argvals)) else 1
  observed_stat <- sum((mu_resid_A - mu_resid_B)^2) * delta
  perm_stats <- numeric(n_perms)
  if (progress) pb <- utils::txtProgressBar(min = 0, max = n_perms, style = 3)
  for (i in seq_len(n_perms)) {
    perm_group_vec <- sample(group_fac)
    perm_mu_A <- colMeans(residuals_mat[perm_group_vec == groups[1], ])
    perm_mu_B <- colMeans(residuals_mat[perm_group_vec == groups[2], ])
    perm_stats[i] <- sum((perm_mu_A - perm_mu_B)^2) * delta
    if (progress) utils::setTxtProgressBar(pb, i)
  }
  if (progress) close(pb)
  p_value <- (sum(perm_stats >= observed_stat) + 1) / (n_perms + 1)
  list(
    passed = p_value >= alpha,
    p_value = p_value,
    statistic = observed_stat
  )
}

#' @title Test for Strict Invariance
#' @description Permutation test to check if the residual error variances
#' are equivalent across groups.
#'
#' @inheritParams test_configural
#' @return A list with `passed` (boolean), `p_value`, and `statistic`.
#' @export
#' @importFrom refund fpca.sc
#'
#' @examples
#' \donttest{
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   res <- test_strict(sim$Y_mat, sim$group_vec, sim$argvals,
#'                      npc = 3, n_perms = 9, alpha = 0.05, progress = FALSE)
#'   print(res$p_value)
#' }
test_strict <- function(Y_mat, group_vec, argvals, npc, n_perms, alpha, progress) {
  if (!is.matrix(Y_mat)) {
    Y_mat <- as.matrix(Y_mat)
  }
  group_fac <- factor(group_vec)
  idx_A <- which(group_fac == levels(group_fac)[1])
  fpca_A <- refund::fpca.sc(Y = Y_mat[idx_A, ], argvals = argvals, npc = npc)
  fpca_B <- refund::fpca.sc(Y = Y_mat[-idx_A, ], argvals = argvals, npc = npc)
  obs_sigma2_A <- fpca_A$sigma2
  obs_sigma2_B <- fpca_B$sigma2
  if (is.null(obs_sigma2_A) || is.null(obs_sigma2_B)) {
    warning("Could not compute sigma2 from original data. Skipping Strict test.")
    return(list(passed = NA, p_value = NA, statistic = NA))
  }
  observed_stat <- abs(obs_sigma2_A - obs_sigma2_B)
  perm_stats <- numeric(n_perms)
  N <- length(group_fac)
  N_A <- length(idx_A)
  if (progress) pb <- utils::txtProgressBar(min = 0, max = n_perms, style = 3)
  for (i in seq_len(n_perms)) {
    perm_ids_A <- sample.int(N, size = N_A, replace = FALSE)
    perm_ids_B <- setdiff(seq_len(N), perm_ids_A)
    fpca_perm_A <- refund::fpca.sc(Y = as.matrix(Y_mat[perm_ids_A, ]), argvals = argvals, npc = npc)
    fpca_perm_B <- refund::fpca.sc(Y = as.matrix(Y_mat[perm_ids_B, ]), argvals = argvals, npc = npc)
    perm_sigma2_A <- fpca_perm_A$sigma2
    perm_sigma2_B <- fpca_perm_B$sigma2
    if (is.null(perm_sigma2_A) || is.null(perm_sigma2_B)) {
      perm_stats[i] <- NA
    } else {
      perm_stats[i] <- abs(perm_sigma2_A - perm_sigma2_B)
    }
    if (progress) utils::setTxtProgressBar(pb, i)
  }
  if (progress) close(pb)
  valid_perms <- sum(!is.na(perm_stats))
  if (valid_perms == 0) {
    p_value <- NA
  } else {
    p_value <- (sum(perm_stats >= observed_stat, na.rm = TRUE) + 1) / (valid_perms + 1)
  }
  list(
    passed = p_value >= alpha,
    p_value = p_value,
    statistic = observed_stat
  )
}

# ============================================================================
# 3. Main Wrapper and Post-Hoc Functions
# ============================================================================

#' @title Run Hierarchical Functional Measurement Invariance (FMI) Tests
#' @description A wrapper function that performs a sequence of permutation tests
#' for configural, metric, scalar, and (optionally) strict invariance.
#'
#' @param Y_mat A numeric matrix (N x M) where N is subjects, M is time points.
#' @param group_vec A vector of length N specifying group membership (2 groups).
#' @param argvals A numeric vector of length M listing the observation points.
#' @param alpha Significance level (default: 0.05).
#' @param npc Optional. The number of FPCs. If NULL (default), it is
#' determined automatically by `determine_npc`.
#' @param target_pve The target PVE if `npc` is NULL (default: 0.95).
#' @param max_npc The max FPCs if `npc` is NULL (default: 6L).
#' @param n_perms The number of permutations (default: 499L).
#' @param strict_test Boolean. Whether to perform the strict invariance test
#' (default: FALSE).
#' @param progress Boolean. Show progress bars? (default: interactive()).
#'
#' @return A list containing test results for each invariance level,
#' settings, and the pooled FPCA object.
#' @export
#' @importFrom refund fpca.sc
#'
#' @examples
#' \donttest{
#'   # Use small N and n_perms for a quick example
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   fmi_results <- run_fmi(
#'     Y_mat = sim$Y_mat,
#'     group_vec = sim$group_vec,
#'     argvals = sim$argvals,
#'     n_perms = 9, # Use 499+ for actual research
#'     progress = FALSE
#'   )
#'  print(fmi_results)
#' }
run_fmi <- function(Y_mat, group_vec, argvals,
                    alpha = 0.05,
                    npc = NULL,
                    target_pve = 0.95,
                    max_npc = 6L,
                    n_perms = 499L,
                    strict_test = FALSE,
                    progress = interactive()) {

  start_time <- proc.time()

  if (!is.matrix(Y_mat)) {
    message("Input Y_mat was coerced to a matrix.")
    Y_mat <- as.matrix(Y_mat)
  }

  group_fac <- factor(group_vec)
  if (nlevels(group_fac) != 2L) {
    stop("FMI testing currently only supports two-group comparisons.")
  }

  # Determine NPC
  npc_info <- if (is.null(npc)) {
    determine_npc(Y_mat, argvals, target_pve, max_npc)
  } else {
    list(
      npc = as.integer(npc),
      pooled_fpca = refund::fpca.sc(Y = Y_mat, argvals = argvals, npc = as.integer(npc)),
      scree = NULL
    )
  }
  npc_val <- npc_info$npc

  results <- list(
    settings = list(alpha = alpha, npc = npc_val, n_perms = n_perms),
    scree_data = npc_info$scree,
    pooled_fpca = npc_info$pooled_fpca
  )

  # Use message() instead of cat() for all console output
  message("====================================================")
  message("  Starting Functional Measurement Invariance (FMI) Tests")
  message(sprintf(" - Number of FPCs (npc): %d", npc_val))
  message(sprintf(" - Number of Permutations: %d", n_perms))
  message("====================================================\n")

  # --- Step 1: Configural ---
  message("--- Step 1: Configural Invariance ---")
  res <- test_configural(Y_mat, group_fac, argvals, npc_val, n_perms, alpha, progress)
  results$configural <- res
  # We still print the plot object as it's a key output
  if (interactive()) {
    print(res$plot)
  }
  message(sprintf("P-value: %.4f. Invariance %s.\n", res$p_value, ifelse(res$passed, "Established", "Rejected")))
  if (!res$passed) {
    end_time <- proc.time()
    message(sprintf("\n>> Total elapsed time: %.2f seconds", (end_time - start_time)["elapsed"]))
    return(results)
  }

  # --- Step 2: Metric ---
  message("--- Step 2: Metric ---")
  res <- test_metric(Y_mat, group_fac, argvals, npc_val, n_perms, alpha, progress)
  results$metric <- res
  message(sprintf("P-value: %.4f. Invariance %s.\n", res$p_value, ifelse(res$passed, "Established", "Rejected")))
  if (!res$passed) {
    end_time <- proc.time()
    message(sprintf("\n>> Total elapsed time: %.2f seconds", (end_time - start_time)["elapsed"]))
    return(results)
  }

  # --- Step 3: Scalar ---
  message("--- Step 3: Scalar Invariance ---")
  res <- test_scalar(Y_mat, group_fac, argvals, npc_info$pooled_fpca, n_perms, alpha, progress)
  results$scalar <- res
  message(sprintf("P-value: %.4f. Invariance %s.\n", res$p_value, ifelse(res$passed, "Established", "Rejected")))
  if (!res$passed) {
    message(">> Scalar invariance rejected. Latent mean comparison is not recommended.")
    end_time <- proc.time()
    message(sprintf("\n>> Total elapsed time: %.2f seconds", (end_time - start_time)["elapsed"]))
    return(results)
  }

  # --- Step 4: Strict (Optional) ---
  if (isTRUE(strict_test)) {
    message("--- Step 4: Strict Invariance ---")
    res <- test_strict(Y_mat, group_fac, argvals, npc_val, n_perms, alpha, progress)
    results$strict <- res
    message(sprintf("P-value: %.4f. Invariance %s.\n", res$p_value, ifelse(res$passed, "Established", "Rejected")))
  }

  message("====================================================")
  message("        FMI Tests Complete (All Levels Passed)")
  message("====================================================")
  message("\n>> All required levels of invariance established.")
  message(">> You may now proceed with compare_latent_means().")

  end_time <- proc.time()
  total_time <- end_time - start_time
  message(sprintf("\n>> Total elapsed time: %.2f seconds", total_time["elapsed"]))

  results$total_elapsed_time <- total_time
  results
}

#' @title Compare Latent Means
#' @description Performs Welch's t-tests on FPC scores between groups.
#' This function should only be run after scalar invariance is established.
#'
#' @param fmi_results The list object returned by `run_fmi()`.
#' @param group_vec The original group vector used in `run_fmi()`.
#'
#' @return A tibble summarizing the t-test results for each FPC.
#' @export
#' @importFrom tibble as_tibble tibble
#' @importFrom stats t.test
#' @importFrom purrr map_dfr
#' @importFrom knitr kable
#'
#' @examples
#' \donttest{
#'   sim <- simulate_fmi_data(N_A = 20, N_B = 20, T_points = 30)
#'   fmi_results <- run_fmi(
#'     Y_mat = sim$Y_mat,
#'     group_vec = sim$group_vec,
#'     argvals = sim$argvals,
#'     n_perms = 9,
#'     progress = FALSE
#'   )
#'   # This will run only if scalar invariance passed
#'   if (!is.null(fmi_results$scalar) && fmi_results$scalar$passed) {
#'     mean_results <- compare_latent_means(fmi_results, sim$group_vec)
#'     print(mean_results)
#'   }
#' }
compare_latent_means <- function(fmi_results, group_vec) {
  if (is.null(fmi_results$scalar) || !fmi_results$scalar$passed) {
    stop("Scalar invariance is not established. Cannot compare latent means.")
  }
  pooled_fpca <- fmi_results$pooled_fpca
  score_df <- tibble::as_tibble(pooled_fpca$scores)
  colnames(score_df) <- paste0("FPC", seq_len(ncol(score_df)))
  score_df$group <- factor(group_vec)
  message("--- Latent Mean Comparison (Welch's t-tests) ---")
  results_table <- purrr::map_dfr(seq_len(pooled_fpca$npc), function(k) {
    comp <- paste0("FPC", k)
    tt <- stats::t.test(score_df[[comp]] ~ score_df$group)
    tibble::tibble(
      component = comp,
      mean_group1 = tt$estimate[1],
      mean_group2 = tt$estimate[2],
      mean_diff = tt$estimate[1] - tt$estimate[2],
      t_stat = tt$statistic,
      p_value = tt$p.value
    )
  })
  # The print() inside the function is okay as it's the main point
  print(knitr::kable(results_table, digits = 3))
  invisible(results_table)
}

# ============================================================================
# 4. Simulation and Example Data Functions
# ============================================================================

#' @title Simulate FMI Data
#' @description Generates functional data for two groups with optional
#' differences in latent means or scalar intercepts.
#'
#' @param N_A Sample size for Group A (default: 50).
#' @param N_B Sample size for Group B (default: 50).
#' @param T_points Number of time points (default: 51).
#' @param mean_shift A shift added to FPC 1 scores for Group B (default: 0).
#' @param scalar_shift A shift added to the mean function for Group B (default: 0).
#' @param eigenvalues A vector of eigenvalues for FPCs (default: c(4, 1, 0.5, 0.1)).
#' @param noise_sd Standard deviation of measurement error (default: 0.5).
#' @param seed Optional. A random seed for reproducibility.
#'
#' @return A list with `Y_mat`, `group_vec`, and `argvals`.
#' @export
#' @importFrom stats rnorm
#' @importFrom methods is
#'
#' @examples
#' sim_data <- simulate_fmi_data(N_A = 10, N_B = 10, T_points = 20, seed = 123)
#' str(sim_data)
simulate_fmi_data <- function(N_A = 50, N_B = 50, T_points = 51,
                              mean_shift = 0, scalar_shift = 0,
                              eigenvalues = c(4, 1, 0.5, 0.1),
                              noise_sd = 0.5, seed = NULL) {
  if (!is.null(seed)) set.seed(seed)
  N <- N_A + N_B
  t <- seq(0, 1, length.out = T_points)
  mu <- 4 * t + cos(2 * pi * t)
  K <- length(eigenvalues)
  phi <- matrix(NA_real_, nrow = K, ncol = T_points)
  phi[1, ] <- sqrt(2) * sin(2 * pi * t)
  phi[2, ] <- sqrt(2) * cos(2 * pi * t)
  if (K >= 3) phi[3, ] <- sqrt(2) * sin(4 * pi * t)
  if (K >= 4) phi[4, ] <- sqrt(2) * cos(4 * pi * t)
  scores <- matrix(NA_real_, nrow = N, ncol = K)
  for (k in seq_len(K)) {
    scores[, k] <- stats::rnorm(N, mean = 0, sd = sqrt(eigenvalues[k]))
  }
  if (mean_shift != 0) {
    scores[(N_A + 1):N, 1] <- scores[(N_A + 1):N, 1] + mean_shift
  }
  mu_mat <- matrix(mu, nrow = N, ncol = T_points, byrow = TRUE)
  if (scalar_shift != 0) {
    mu_mat[(N_A + 1):N, ] <- mu_mat[(N_A + 1):N, ] + scalar_shift
  }
  error_mat <- matrix(stats::rnorm(N * T_points, mean = 0, sd = noise_sd), nrow = N)
  Y_mat <- mu_mat + scores %*% phi + error_mat
  group_vec <- factor(rep(c("A", "B"), times = c(N_A, N_B)))
  list(Y_mat = Y_mat, group_vec = group_vec, argvals = t)
}

#' @title DTI Example Data Wrapper
#' @description A helper function to load and pre-process the DTI dataset
#' from the 'refund' package for use in examples.
#'
#' @param n_perms The number of permutations (default: 499L).
#'        Passed to `run_fmi`.
#'
#' @return Invisibly returns the FMI results list.
#' @export
#' @importFrom utils data
#' @importFrom dplyr select filter
#' @importFrom tibble tibble as_tibble
#' @importFrom tidyr drop_na
#' @importFrom stats na.omit
#'
#' @examples
#' \donttest{
#'   # n_perms=9 is for a quick check. Use 499+ for real analysis.
#'   if (requireNamespace("refund", quietly = TRUE)) {
#'     dti_results <- run_dti_example(n_perms = 9)
#'   }
#' }
run_dti_example <- function(n_perms = 499L) {
  if (!requireNamespace("refund", quietly = TRUE)) {
    message("Package 'refund' needed for this example to run.")
    return(invisible(NULL))
  }
  utils::data("DTI", package = "refund")
  dti_meta_data <- tibble::tibble(
    ID = DTI$ID,
    visit = DTI$visit,
    sex = DTI$sex,
    case = factor(DTI$case, levels = c(0, 1), labels = c("Control", "Patient"))
  )
  dti_cca_data <- tibble::as_tibble(DTI$cca) %>%
    stats::setNames(paste0("cca_", 1:ncol(DTI$cca))) # Use stats::setNames
  dti_filtered_df <- dplyr::bind_cols(dti_meta_data, dti_cca_data) %>%
    dplyr::filter(.data$visit == 1) %>%
    dplyr::select(.data$ID, .data$case, .data$sex, dplyr::starts_with("cca_"))
  if (requireNamespace("tidyr", quietly = TRUE)) {
    dti_filtered_df <- tidyr::drop_na(dti_filtered_df, dplyr::starts_with("cca_"))
  } else {
    dti_filtered_df <- stats::na.omit(dti_filtered_df)
  }
  if (nrow(dti_filtered_df) == 0) {
    stop("No valid (non-NA) data found for visit == 1 in DTI example.")
  }
  Y_mat <- dti_filtered_df %>%
    dplyr::select(dplyr::starts_with("cca_")) %>%
    as.matrix()
  group_vec <- factor(dti_filtered_df$case)
  argvals <- seq(0, 1, length.out = ncol(Y_mat))
  message("\n\n===== Real Data Example: DTI (Control vs Patient) =====\n")
  fmi_results <- run_fmi(Y_mat, group_vec, argvals, n_perms = n_perms, progress = TRUE, strict_test=TRUE)
  if (!is.null(fmi_results$scalar) && fmi_results$scalar$passed) {
    compare_latent_means(fmi_results, group_vec)
  }
  invisible(fmi_results)
}

#' @title Berkeley Growth Example Data Wrapper
#' @description A helper function to load and pre-process the Growth dataset
#' from the 'fda' package for use in examples.
#'
#' @param n_perms The number of permutations (default: 499L).
#'        Passed to `run_fmi`.
#'
#' @return Invisibly returns the FMI results list.
#' @export
#' @importFrom utils data
#'
#' @examples
#' \donttest{
#'   # n_perms=9 is for a quick check. Use 499+ for real analysis.
#'   if (requireNamespace("fda", quietly = TRUE)) {
#'     growth_results <- run_growth_example(n_perms = 9)
#'   }
#' }
run_growth_example <- function(n_perms = 499L) {
  if (!requireNamespace("fda", quietly = TRUE)) {
    message("Package 'fda' needed for this example to run.")
    return(invisible(NULL))
  }
  utils::data("growth", package = "fda")
  Y_boys <- t(growth$hgtm)
  Y_girls <- t(growth$hgtf)
  Y_mat <- rbind(Y_boys, Y_girls)
  group_vec <- factor(
    rep(c("Boys", "Girls"), times = c(nrow(Y_boys), nrow(Y_girls)))
  )
  argvals <- growth$age
  message("\n\n===== Real Data Example: Berkeley Growth (Boys vs Girls) =====\n")
  fmi_results <- run_fmi(Y_mat, group_vec, argvals, n_perms = n_perms, progress = TRUE)
  if (!is.null(fmi_results$scalar) && fmi_results$scalar$passed) {
    compare_latent_means(fmi_results, group_vec)
  }
  invisible(fmi_results)
}
