% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/select_mtpi_mtd.R
\name{select_mtpi_mtd}
\alias{select_mtpi_mtd}
\title{Select dose by mTPI's MTD-choosing algorithm.}
\usage{
select_mtpi_mtd(
  parent_selector_factory,
  when = c("finally", "always"),
  target = NULL,
  exclusion_certainty,
  alpha = 1,
  beta = 1,
  pava_just_tested_doses = FALSE,
  ...
)
}
\arguments{
\item{parent_selector_factory}{Object of type \code{\link{selector_factory}}.}

\item{when}{Either of: 'finally' to select dose only when the parent
dose-selector has finished, by returning continue() == FALSE; or 'always'
to use this dose-selection algorithm for every dose decision. As per the
authors' original intentions, the default is 'finally'.}

\item{target}{We seek a dose with this probability of toxicity. If not
provided, the value will be sought from the parent dose-selector.}

\item{exclusion_certainty}{Numeric, threshold posterior certainty required to
exclude a dose for being excessively toxic. The authors discuss values in the
range 0.7 - 0.95. Set to a value > 1 to suppress the dose exclusion
mechanism. The authors use the Greek letter xi for this parameter.}

\item{alpha}{First shape parameter of the beta prior distribution on the
probability of toxicity.}

\item{beta}{Second shape parameter of the beta prior distribution on the
probability of toxicity.}

\item{pava_just_tested_doses}{the design uses the PAVA method to estimate
monotonic Prob(Tox) at the doses. By default, this estimates Prob(Tox) for
doses that have not been tested in the trial and can lead to untested
doses being recommended. Set this option to TRUE to use PAVA only on tested
doses, leaving untested doses as NA and not-recommendable. Set to FALSE to
use PAVA at all doses and potentially recommend an untested dose. Default
is FALSE.}

\item{...}{Extra args are passed onwards.}
}
\value{
an object of type \code{\link{selector_factory}}.
}
\description{
Note: if you use this selector, it almost certainly needs to be the last
example in the chain - see Example below. This method selects dose by the
algorithm for identifying the maximum tolerable dose (MTD) described in Ji et
al. (2010). This class is intended to be used when a mTPI trial has reached
its maximum sample size. Thus, it intends to make the final dose
recommendation after the regular mTPI dose selection algorithm, as
implemented by \code{\link{get_mtpi}}, including any additional behaviours
that govern stopping (etc), has gracefully concluded a dose-finding trial.
However, the class can be used in any scenario where there is a target
toxicity rate. See Examples. Note - this class will not override the parent
dose selector when the parent is advocating no dose. Thus this class will not
reinstate a dangerous dose.
}
\examples{
# This class is intended to make the final dose selection in a mTPI trial:
target <- 0.25
model <- get_mtpi(num_doses = 5, target = target,
                  epsilon1 = 0.05, epsilon2 = 0.05,
                  exclusion_certainty = 0.95) \%>\%
  stop_at_n(n = 12) \%>\%
  select_mtpi_mtd(exclusion_certainty = 0.95)

outcomes <- '1NNN 2NTN 2NNN 3NTT'
model \%>\% fit(outcomes) \%>\% recommended_dose()

# However, since behaviour is modular in this package, we can use this method
# to select dose at every dose decision if we wanted:
model2 <- get_mtpi(num_doses = 5, target = target,
                   epsilon1 = 0.05, epsilon2 = 0.05,
                   exclusion_certainty = 0.95) \%>\%
  select_mtpi_mtd(when = 'always', exclusion_certainty = 0.95)
model2 \%>\% fit('1NNT') \%>\% recommended_dose()
model2 \%>\% fit('1NNN 2NNT') \%>\% recommended_dose()

# and with any underlying model:
skeleton <- c(0.05, 0.1, 0.25, 0.4, 0.6)
model3 <- get_dfcrm(skeleton = skeleton, target = target) \%>\%
  select_mtpi_mtd(when = 'always', exclusion_certainty = 0.95)
model3 \%>\% fit('1NNT') \%>\% recommended_dose()
model3 \%>\% fit('1NNN 2NNT') \%>\% recommended_dose()

}
\references{
Ji, Y., Liu, P., Li, Y., & Bekele, B. N. (2010). A modified
toxicity probability interval method for dose-finding trials. Clinical
Trials, 7(6), 653-663. https://doi.org/10.1177/1740774510382799

Ji, Y., & Yang, S. (2017). On the Interval-Based Dose-Finding Designs, 1-26.
Retrieved from https://arxiv.org/abs/1706.03277
}
