% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils_cd.R
\name{cd_test}
\alias{cd_test}
\alias{cd_test.default}
\alias{cd_test.csdm_fit}
\alias{print.cd_test}
\title{Cross-sectional dependence (CD) tests for panel residuals}
\usage{
cd_test(object, ...)

\method{cd_test}{default}(
  object,
  type = c("CD", "CDw", "CDw+", "CDstar", "all"),
  n_pc = 4L,
  seed = NULL,
  min_overlap = 2L,
  na.action = c("drop.incomplete.times", "pairwise"),
  ...
)

\method{cd_test}{csdm_fit}(
  object,
  type = c("CD", "CDw", "CDw+", "CDstar", "all"),
  n_pc = 4L,
  seed = NULL,
  min_overlap = 2L,
  na.action = c("drop.incomplete.times", "pairwise"),
  ...
)

\method{print}{cd_test}(x, digits = 3, ...)
}
\arguments{
\item{object}{A \code{csdm_fit} model object or a numeric matrix of residuals (N x T).}

\item{...}{Additional arguments passed to methods.}

\item{type}{Which test(s) to compute: one of \code{"CD"}, \code{"CDw"}, \code{"CDw+"},
\code{"CDstar"}, or \code{"all"} (default: \code{"CD"}).}

\item{n_pc}{Number of principal components for CD* (default 4).}

\item{seed}{Integer seed for weight draws in CDw/CDw+ (default NULL = no seed set).}

\item{min_overlap}{Minimum number of overlapping time periods required for a unit
pair to be included in CD/CDw/CDw+ (default 2).}

\item{na.action}{How to handle missing data: \code{"drop.incomplete.times"} (default)
removes time periods with any missing observations to create a balanced panel for CD*;
\code{"pairwise"} uses pairwise correlations for CD/CDw/CDw+ and warns for CD*.}

\item{x}{An object of class \code{cd_test}.}

\item{digits}{Number of digits to print (default 3).}
}
\value{
An object of class \code{cd_test} with fields \code{tests}, \code{type},
\code{N}, \code{T}, \code{na.action}, and \code{call}. The \code{tests} list
contains one or more test results, each with \code{statistic} and \code{p.value}.
}
\description{
Computes Pesaran CD, CDw, CDw+, and CD* tests for cross-sectional dependence
in panel residuals. The implementation supports residual matrices or fitted
\code{csdm_fit} objects and provides consistent handling of unbalanced panels.
}
\details{
\subsection{Notation}{

Let \eqn{E} be the residual matrix with \eqn{N} cross-sectional units and \eqn{T}
time periods. For each unit pair \eqn{(i,j)}, let \eqn{T_{ij}} be the number of
overlapping time periods and \eqn{\rho_{ij}} the pairwise correlation.
}

\subsection{Test statistics}{

\describe{
\item{CD (Pesaran, 2015)}{
\deqn{CD = \sqrt{\frac{2}{N(N-1)}} \sum_{i<j} \sqrt{T_{ij}} \, \rho_{ij}}
}
\item{CDw (Juodis and Reese, 2021)}{
Random sign flips \eqn{w_i \in \{-1,1\}} are applied to residuals before
computing correlations. The statistic is CD applied to the sign-flipped data.
}
\item{CDw+ (Fan, Liao, and Yao, 2015)}{
Power enhancement adds a sparse thresholding term to CDw. The threshold is
\deqn{c_N = \sqrt{\frac{2 \log(N)}{T}}}
and the power term sums \eqn{\sqrt{T_{ij}} |\rho_{ij}|} for pairs exceeding
the threshold.
}
\item{CD* (Pesaran and Xie, 2021)}{
CD is computed on residuals after removing \code{n_pc} principal components
from \eqn{E}. This provides a bias-corrected test under multifactor errors.
}
}
}

\subsection{Missing data and balance}{

\describe{
\item{CD, CDw, CDw+}{Always use pairwise-complete observations. Each pairwise
correlation uses available overlaps.}
\item{CD*}{Requires a balanced panel. By default, \code{na.action = "drop.incomplete.times"}
removes any time period with missing observations. With \code{na.action = "pairwise"},
CD* returns \code{NA} and a warning when missing values are present.}
}
}
}
\examples{
# Simulate independent and dependent panels
set.seed(1)
E_indep <- matrix(rnorm(100), nrow = 10)
E_dep <- matrix(rnorm(10), nrow = 10, ncol = 10, byrow = TRUE)

# Compute all tests
cd_test(E_indep, type = "all")
cd_test(E_dep, type = "all")

# Specific test with parameters
cd_test(E_indep, type = "CDstar", n_pc = 2)

# From a fitted csdm model
data(PWT_60_07, package = "csdm")
df <- PWT_60_07
ids <- unique(df$id)[1:10]
df_small <- df[df$id \%in\% ids & df$year >= 1970, ]
fit <- csdm(
  log_rgdpo ~ log_hc + log_ck + log_ngd,
  data = df_small,
  id = "id",
  time = "year",
  model = "cce",
  csa = csdm_csa(vars = c("log_rgdpo", "log_hc", "log_ck", "log_ngd"))
)
cd_test(fit, type = "all")

}
\references{
Pesaran, M.H. (2015). "Testing weak cross-sectional dependence in large panels."
\emph{Econometric Reviews}, 34(6-10), 1089-1117.

Pesaran, M.H. (2021). "General diagnostic tests for cross-sectional dependence
in panels." \emph{Empirical Economics}, 60, 13-50.

Juodis, A., & Reese, S. (2021). "The incidental parameters problem in testing for
remaining cross-sectional correlation." \emph{Journal of Business and Economic Statistics},
40(3), 1193-1203.

Fan, J., Liao, Y., & Yao, J. (2015). "Power Enhancement in High-Dimensional
Cross-Sectional Tests." \emph{Econometric Reviews}, 34(6-10), 742-779.

Pesaran, M.H., & Xie, Y. (2021). "A bias-corrected CD test for error cross-sectional
dependence in panel models." \emph{Econometric Reviews}, 41(6), 649-677.
}
