#' Compute Confidence Interval for Variance Without Raw Data
#'
#' This function calculates the confidence interval for variance when only + the sample variance and sample size are known.
#'
#' @param s2_given Numeric. The sample variance.
#' @param n_given Integer. The sample size.
#' @param conf_level Numeric. The confidence level (e.g., 0.95 for 95% CI).
#'
#' @return A numeric vector with the lower and upper bounds of the confidence interval.
#'
#' @examples
#' s2_given <- 21.5   # Sample variance
#' n_given <- 26      # Sample size
#' conf_level <- 0.98 # 98% confidence level
#' chi_var_ci_no_data(s2_given, n_given, conf_level)
#' @importFrom stats qchisq var
#'
#' @export
chi_var_ci_no_data <- function(s2_given, n_given, conf_level) {
  # Check inputs
  if (!is.numeric(s2_given) || !is.numeric(n_given) || !is.numeric(conf_level)) {
    stop("All inputs must be numeric")
  }

  # Compute chi-square confidence interval
  alpha <- 1 - conf_level
  chi_sq_lower <- qchisq(1 - alpha/2, df = n_given - 1)
  chi_sq_upper <- qchisq(alpha/2, df = n_given - 1)

  lower_bound <- ((n_given - 1) * s2_given) / chi_sq_lower
  upper_bound <- ((n_given - 1) * s2_given) / chi_sq_upper

  return(c(lower_bound, upper_bound))
}
