library(testthat)
library(RealSurvSim)


# Sample dataset creation
set.seed(123)
#data("liang_recon", package = "RealSurvSim")

dat <- dats$Liang
test_that("Test 1: n parameter works correctly", {
  result_with_default_n <- data_simul_Bootstr(dat)
  result_with_custom_n <- data_simul_Bootstr(dat, n = 50 ,type = "case")

  expect_equal(nrow(result_with_default_n), nrow(dat)) # Should be the same as original data
  expect_equal(nrow(result_with_custom_n), 50) # Custom number of simulations
})

test_that("Test 2: bootstrap type 'case' and 'cond' works correctly", {
  result_case <- data_simul_Bootstr(dat, type = "case")
  result_cond <- data_simul_Bootstr(dat, type = "cond")

  expect_s3_class(result_case, "data.frame")
  expect_s3_class(result_cond, "data.frame")
  expect_equal(ncol(result_case), ncol(dat))
  expect_equal(ncol(result_cond), 2)
})

test_that("Test 3: handles ties in event and censoring times", {
  result_cond <- data_simul_Bootstr(dat, type = "cond")
  original_events <- dat$V1[dat$V2 == 1]
  original_censored <- dat$V1[dat$V2 == 0]

  # Check if any censored time was adjusted due to a tie with an event time
  adjusted_censored <- result_cond$V1[dat$V2 == 0]
  expect_true(any(adjusted_censored != original_censored))
})

test_that("Test 4: throws error for unsupported bootstrap types", {
  expect_error(data_simul_Bootstr(dat, type = "unknown"), "This bootstrap method is not implemented.")
})

test_that("Test 5: handles NULL n parameter correctly", {
  result_with_null_n <- data_simul_Bootstr(dat, n = NULL)
  expect_equal(nrow(result_with_null_n), nrow(dat))
})

test_that("Test 6: output structure and types are correct", {
  result_case <- data_simul_Bootstr(dat, type = "case")
  result_cond <- data_simul_Bootstr(dat, type = "cond")

  expect_true(all(c("V1", "V2") %in% colnames(result_case)))
  expect_true(all(c("V1", "V2") %in% colnames(result_cond)))
  expect_type(result_case$V1, "double")
  expect_type(result_case$V2, "integer")
  expect_type(result_cond$V1, "double")
  expect_type(result_cond$V2, "double")
})


test_that("Test 7: reproducibility of results with set seed", {
  set.seed(123)
  result_1 <- data_simul_Bootstr(dat, n = 50, type = "case")
  set.seed(123)
  result_2 <- data_simul_Bootstr(dat, n = 50, type = "case")
  expect_identical(result_1, result_2)
})

test_that("Test 8: handles empty dataframe", {
  empty_dat <- dat[0, ]
  expect_error(data_simul_Bootstr(empty_dat,n=50, type = "case"), "The dataset is empty.")
  expect_error(data_simul_Bootstr(empty_dat,n=50, type = "cond"), "The dataset is empty.")
})

test_that("Test 9: handles dataframe with only events", {
  dat <- as.data.frame( dat)
  only_events_dat <- dat[dat$V2 == 1, ]

  expect_error(data_simul_Bootstr(only_events_dat, type = "cond"), "No censoring data available for conditional bootstrap.")
})

test_that("Test 10: handles dataframe with only censored", {
  dat <- dat
  only_censored_dat <- dat[dat$V2 == 0, ]

  expect_error(data_simul_Bootstr(only_censored_dat, type = "cond"), "No event data available for conditional bootstrap.")

})


# All tests completed
#cat("All tests for data_simul_Bootstr function executed successfully with the liang_recon dataset.\n")
