#' Simulate Colorblind Vision
#'
#' Shows how your plot appears to people with different types of color vision deficiency
#'
#' @param plot A ggplot2 object
#' @param type Type of CVD: "deutan" (green-blind), "protan" (red-blind), or "tritan" (blue-blind)
#' @return A modified ggplot2 object showing the simulated view
#' @export
#' @examples
#' library(ggplot2)
#' p <- ggplot(mtcars, aes(wt, mpg, color = factor(cyl))) +
#'   geom_point() +
#'   scale_color_manual(values = c("red", "green", "blue"))
#' gg_simulate_cvd(p, type = "deutan")
gg_simulate_cvd <- function(plot, type = "deutan") {

  if (!inherits(plot, "gg")) {
    stop("Input must be a ggplot2 object")
  }

  # Validate type
  valid_types <- c("deutan", "protan", "tritan")
  if (!type %in% valid_types) {
    stop("type must be one of: ", paste(valid_types, collapse = ", "))
  }

  # Build the plot to get the data
  built <- ggplot2::ggplot_build(plot)

  # Extract all colors from the plot
  all_colors <- unique(unlist(lapply(built$data, function(layer) {
    c(layer$colour, layer$fill)
  })))
  all_colors <- all_colors[!is.na(all_colors)]

  if (length(all_colors) == 0) {
    message("No colors found in plot to simulate")
    return(plot)
  }

  # Simulate colorblind vision using colorspace
  simulated_colors <- colorspace::deutan(all_colors, severity = 1)
  if (type == "protan") {
    simulated_colors <- colorspace::protan(all_colors, severity = 1)
  } else if (type == "tritan") {
    simulated_colors <- colorspace::tritan(all_colors, severity = 1)
  }

  message("Simulating ", type, " color vision deficiency...")
  message("Original colors: ", paste(all_colors, collapse = ", "))
  message("Simulated colors: ", paste(simulated_colors, collapse = ", "))

  # Add subtitle to indicate simulation
  plot + ggplot2::labs(
    subtitle = paste0("Simulated ", type, "anopia view"),
    caption = "Generated by GGenemy package"
  )
}
