##' @title Visualization of the sequence (genotypes) data
##'
##' @description Display the sequence data, such as the data generated by simX. The horizontal axis represents the sampling sites, and the vertical axis represents the sequence values containing only 0, 1 and 2.
##'
##' @param location a numeric vector defining the sampling sites of the sequence data.
##' @param X a matrix specifying the sequence data, with the number of rows equal to the number of samples.
##' @param color A specification for the default plotting color. See graphical parameters (\code{\link{par}}) for more details.
##' @param pch Either an integer specifying a symbol or a single character to be used as the default in plotting points. See graphical parameters (\code{\link{par}}) for more details.
##' @param cex A numerical value giving the amount by which plotting text and symbols should be magnified relative to the default. See graphical parameters (\code{\link{par}}) for more details.
##'
##' @return show the sequence data.
##'
##' @seealso See Also as \code{\link{simX}}, \code{\link{SNPgvf}}.
##'
##' @importFrom funData funData
##' @importFrom funData irregFunData
##' @importFrom funData plot
##' @export
##' @examples
##' library(FunctanSNP)
##' n <- 20
##' m <- 50
##' simdata <- simX(n, m, seed = 1, d.ratio = 0)
##' X <- simdata$X
##' location <- simdata$location
##' plotRawdata(location, X = X[1:2, ])
##'

#library(funData)
plotRawdata <- function(location, X, color = NULL, pch = 16, cex = 0.9){
  if (!inherits(location, "numeric")) { stop("x should be of vector type.") }
  if (!inherits(X, "matrix")) { stop("X should be of matrix type.") }

  n <- nrow(X)
  m <- ncol(X)
  type = 'o'
  truelengths <- length( which( is.na(as.vector(X)) == FALSE ) )
  if(truelengths == n*m){#Regular
    fundata <- funData(argvals = location, X = X)
    if(is.null(color) == TRUE){ plot(fundata, type = type, pch = pch, cex = cex, xlab = "Location", ylab = "X") }
    if(is.null(color) == FALSE){ plot(fundata, col = color, type = type, pch = pch, cex = cex, xlab = "Location", ylab = "X")}
  }

  if(truelengths != n*m){#Irregular
    location_list <- lapply(1:n, function(i) location[which(is.na(X[i, ]) == FALSE)])
    X_list <- lapply(1:n, function(i) X[i, which(is.na(X[i, ]) == FALSE)])
    fundata <- irregFunData(argvals = location_list, X = X_list)
    if(is.null(color) == TRUE){ plot(fundata, type = type, pch = pch, cex = cex, xlab = "Location", ylab = "X") }
    if(is.null(color) == FALSE){ plot(fundata, col = color, type = type, pch = pch, cex = cex, xlab = "Location", ylab = "X")}
  }
}
