#' @importClassesFrom tidytree treedata
setClass("fbdrange", contains = "treedata")
setGeneric("as.fbdrange", function(data, ...) standardGeneric("as.fbdrange"))

#' Creates an fbdrange object from provided tree and data
#'
#' @param tree \code{phylo} object to be included in the fbdrange object
#' @param data associated data to be included in the fbdrange object. Needs to contain \code{range} and \code{orientation} fields.
#'
#' @return the new fbdrange object
#' @export
fbdrange = function(tree, data) {
  obj = tidytree::treedata(phylo = tree, data = data)
  as.fbdrange.treedata(obj)
}

#' @importFrom methods new
as.fbdrange.treedata = function(data, ...) {
  if(is.null(data@data$range) || is.null(data@data$orientation)) stop("Fbdrange objects require range and orientation data")
  new("fbdrange", data)
}
setMethod("as.fbdrange", "treedata", as.fbdrange.treedata)

#' Import fbdrange object from file
#'
#' @param input_file path to a tree file in Nexus format containing range and orientation metadata for the trees (for instance, a file generated by the BEAST2 package sRanges)
#'
#' @return an fbdrange object
#' @export
#'
#' @examples
#' tree_file <- system.file("extdata", "fbdrange.trees", package = "FossilSim")
#' fbdr <- get_fbdrange_from_file(tree_file)
get_fbdrange_from_file = function(input_file) {
  if (!requireNamespace("treeio", quietly = TRUE)) {
    stop("Package treeio is needed for this function to work. Please install it.", call. = FALSE)
  }
  
  tree <- treeio::read.beast(input_file)
  as.fbdrange(tree)
}
