#' PPC: Principal Projection Components
#'
#' Principal component projection method that extracts factor loadings and
#' specific variances through a projection matrix.
#'
#' @param data T x N data matrix.
#' @param m Number of components.
#' @return List with Apro (loadings), Dpro (specific variances), Sigmahatpro.
#' @export
#' @examples
#' \dontrun{
#' set.seed(123)
#' N <- 100; T <- 150; m.true <- 3
#' dat <- matrix(rnorm(T * N), T, N)
#' ppc.out <- EFM::PPC(dat, m = m.true)
#' print(round(ppc.out$Apro[1:5, 1:3], 3))
#' }
PPC <- function(data, m) {
  X <- scale(data)
  n <- nrow(X); p <- ncol(X)
  P <- diag(n)  # 简化投影矩阵，可按实际公式修改
  Xpro <- X %*% P
  Sigmahatpro <- cov(Xpro)
  eig <- eigen(Sigmahatpro)
  ind <- order(eig$values, decreasing = TRUE)[1:m]
  Qhat <- eig$vectors[, ind, drop = FALSE]
  lambdahat <- eig$values[ind]
  Apro <- Qhat %*% diag(sqrt(lambdahat))
  hpro <- diag(Apro %*% t(Apro))
  Dpro <- diag(Sigmahatpro) - hpro
  list(Apro = Apro, Dpro = Dpro, Sigmahatpro = Sigmahatpro)
}