test_that("rule_value() works", {
  expect_equal(rule_value("3"), 3)
  expect_equal(rule_value("9"), 9)
  expect_equal(rule_value(":"), 10)
  expect_equal(rule_value("z"), 74)
  expect_error(rule_value(5))
})

test_that("dexi_bool() works", {
  expect_false(dexi_bool(NULL))
  expect_false(dexi_bool(NA))
  expect_false(dexi_bool(""))
  expect_false(dexi_bool(list()))
  expect_false(dexi_bool(0))
  expect_false(dexi_bool("F"))
  expect_false(dexi_bool("tru"))

  expect_true(dexi_bool("true"))
  expect_true(dexi_bool("TRUE"))
  expect_true(dexi_bool("True"))
  expect_true(dexi_bool('t'))
  expect_true(dexi_bool('T'))
  expect_true(dexi_bool(1))
})

test_that("dexi_vector() works", {
  expect_error(dexi_vector(NULL))
  expect_error(dexi_vector(NA))
  expect_error(dexi_vector(list()))
  expect_error(dexi_vector(1))
  expect_error(dexi_vector(c(1, 2)))
  expect_error(dexi_vector(c("a", "b")))

  expect_warning(dexi_vector("a"))
  expect_warning(dexi_vector("1.1;5.5;-7.7xs"))
  expect_warning(expect_identical(dexi_vector("1.1;5.5;-7.7xs"), c(1.1,5.5,NA)))

  expect_equal(dexi_vector(""), numeric(0))
  expect_equal(dexi_vector("1"), 1)
  expect_equal(dexi_vector("1.1"), 1.1)
  expect_equal(dexi_vector("1;2"), c(1,2))
  expect_equal(dexi_vector("1;5;-7"), c(1,5,-7))
  expect_equal(dexi_vector("1.1;5.5;-7.7"), c(1.1,5.5,-7.7))
})

test_that("dexi_value() works", {
  expect_null(dexi_value(NULL))

  expect_error(dexi_value(NA))
  expect_error(dexi_value(list()))
  expect_error(dexi_value(1))
  expect_error(dexi_value(c(1, 2)))
  expect_error(dexi_value(c("a", "b")))
  expect_error(dexi_value("{1.1;2.2}"))
  expect_warning(dexi_value("a"))
  expect_warning(dexi_value("1.1;5.5;-7.7"))

  expect_null(dexi_value(""))
  expect_identical(dexi_value("*"), "*")
  expect_identical(dexi_value("Undefined"), NA)
  expect_equal(dexi_value("1", add = 0), 1)
  expect_equal(dexi_value("1", add = 1), 2)
  expect_equal(dexi_value("1:3"), 1:3)
  expect_equal(dexi_value("<1.1>"), distribution(1.1))
  expect_equal(dexi_value("<1.1;2.2>"), distribution(c(1.1, 2.2)))
  expect_equal(dexi_value("{1;2}"), c(1,2))
  expect_equal(dexi_value("{1;2}", add = 1), c(2,3))
  expect_equal(dexi_value("{1; 5; -7}"), c(1, 5, -7))
  expect_equal(dexi_value("{1; 5; -7}", add = 1), c(2, 6, -6))
  expect_equal(dexi_value("<1.1; 5.5 ;-7.7>"), distribution(c(1.1,5.5,-7.7)))
})

test_that("dexi_option_value() works", {
  expect_null(dexi_option_value(NULL))

  expect_identical(dexi_option_value(NA), NA)
  expect_identical(dexi_option_value(""), "*")
  expect_identical(dexi_option_value("*"), "*")
  expect_identical(dexi_option_value("Undefined"), NA)
  expect_equal(dexi_option_value("123"), c(2, 3, 4))
  expect_identical(dexi_option_value(list()), NA)
  expect_identical(dexi_option_value(1), NA)
  expect_identical(dexi_option_value(c(1, 2)), NA)
  expect_identical(dexi_option_value(c("a", "b")), NA)
})

test_that("distribution() works", {
  expect_error(distribution())
  expect_error(distribution(NULL))
  expect_error(distribution(NA))
  expect_error(distribution("null"))
  expect_true(2 == distribution(2))
  expect_false(inherits(2, DistributionClass))
  expect_true(inherits(distribution(2), DistributionClass))
  expect_true(class(distribution(2)) ==  DistributionClass)

  expect_error(distribution(NULL))
  expect_error(distribution(NA))
  expect_error(distribution("null"))
  expect_equal(distribution(2), distribution(2))
  expect_equal(distribution(2, 3), distribution(c(2, 3)))
})

test_that("is_distribution() works", {
  expect_false(is_distribution(NULL))
  expect_false(is_distribution(NA))
  expect_false(is_distribution(3))
  expect_false(is_distribution("text"))
  expect_false(is_distribution(c(1,2,3)))
  expect_false(is_distribution(CarDxi))
  expect_error(is_distribution(distribution()))
  expect_true(is_distribution(distribution(1,0,2)))
})

test_that("set_to_distr() works", {
  expect_identical(set_to_distr(distribution(1,2)), distribution(1,2))
  expect_error(set_to_distr(NULL))
  expect_error(set_to_distr("x"))
  expect_error(set_to_distr(c("1", "2")))

  expect_equal(set_to_distr(c(1, 3, 4)), distribution(1, 0, 1, 1))
  expect_equal(set_to_distr(c(0, 3, -1, 4)), distribution(0, 0, 1, 1))
  expect_equal(set_to_distr(c(-1, -2, -4)), NA)
})

test_that("distr_to_set() works", {
  expect_equal(distr_to_set("x"), "x")
  expect_null(distr_to_set(NULL))

  expect_equal(distr_to_set(distribution(0, 1, 0, 0.5)), c(2, 4))
  expect_equal(distr_to_set(distribution(0, -1, 1, 0)), 3)
  expect_equal(distr_to_set(distribution(0, -1, 1, 0.5)), c(3, 4))
  expect_equal(distr_to_set(distribution(1, 0.1, 0.2, 0.005), eps = 0.1), c(1, 3))
})

test_that("value_to_set() works", {
  scl <- DexiDiscreteScale(values = c("low", "med", "high"))

  expect_equal(value_to_set(NULL, scl), NA)
  expect_equal(value_to_set(NA, scl), NA)
  expect_equal(value_to_set(c(NA), scl), NA)
  expect_equal(value_to_set(1.2, scl), NA)
  expect_equal(value_to_set(c(1, 1.2), scl), NA)

  expect_equal(value_to_set(1, scl), 1)
  expect_equal(value_to_set(1:2, scl), c(1,2))
  expect_equal(value_to_set(c(1,3), scl), c(1,3))
  expect_equal(value_to_set(distribution(1, 0.5, 0.5), scl), c(1,2,3))
  expect_equal(value_to_set(distribution(1, 0, 0.5), scl), c(1,3))
})

test_that("scale_value() works for continuous scales", {
  scl <- DexiContinuousScale()

  expect_error(scale_value(1, NULL))
  expect_error(scale_value(1, 2))
  expect_error(scale_value(scl, scl))
  expect_error(scale_value(c(1,2), scl))
  expect_error(scale_value(c(NA, 1), scl))
  expect_error(scale_value(c(NA, NA), scl))
  expect_error(scale_value("a", scl))
  expect_error(scale_value(list(1.1), scl))
  expect_error(scale_value(distribution(c(1.1, 2.2)), scl))

  expect_equal(scale_value(NULL, scl), NA)
  expect_equal(scale_value(c(), scl), NA)
  expect_equal(scale_value(list(), scl), NA)
  expect_equal(scale_value(character(), scl), NA)
  expect_equal(scale_value(NA, scl), NA)
  expect_equal(scale_value(c(NA), scl), NA)
  expect_equal(scale_value(15.5, scl), 15.5)
  expect_equal(scale_value(distribution(15.5), scl), 15.5)
})

test_that("scale_value() works for discrete scales", {
  scl <- DexiDiscreteScale(values = c("low", "med", "high"))

  expect_error(scale_value(1, NULL))
  expect_error(scale_value(1, 2))
  expect_error(scale_value(scl, scl))
  expect_error(scale_value(c(NA, 1), scl))
  expect_error(scale_value(c(NA, NA), scl))
  expect_error(scale_value("lo", scl))
  expect_error(scale_value(c("a", "high"), scl))
  expect_error(scale_value(list("a", "high"), scl))

  expect_equal(scale_value(NULL, scl), NA)
  expect_equal(scale_value(c(), scl), NA)
  expect_equal(scale_value(list(), scl), NA)
  expect_equal(scale_value(NA, scl), NA)
  expect_equal(scale_value("*", scl), 1:3)
  expect_equal(scale_value("Undefined", scl), 1:3)
  expect_equal(scale_value(2, scl), 2)
  expect_equal(scale_value(c(-1, 2, 4), scl), c(-1, 2, 4))
  expect_equal(scale_value(distribution(c(-1, 2, 4)), scl), distribution(c(-1, 2, 4)))
  expect_equal(scale_value(c(-1, 2.2, 4), scl), distribution(c(-1, 2.2, 4)))
  expect_equal(scale_value("high", scl), 3)
  expect_equal(scale_value(c("low", "high"), scl), c(1,3))
  v <- c(0.5, 0.4)
  names(v) <- c("low", "a")
  expect_error(scale_value(v, scl))
  names(v) <- c("low", "high")
  expect_equal(scale_value(v, scl), distribution(c(0.5, 0, 0.4)))

  expect_error(scale_value(list(a = 1.1, low = 2.2), scl))
  expect_equal(scale_value(list(high = 1.1, low = 2.2), scl), distribution(c(2.2, 0, 1.1)))
})

test_that("bounded_scale_value() works for continuous scales", {
  scl <- DexiContinuousScale()

  expect_error(bounded_scale_value(1, NULL))
  expect_error(bounded_scale_value(1, 2))
  expect_error(bounded_scale_value(scl, scl))
  expect_error(bounded_scale_value(c(1,2), scl))
  expect_error(bounded_scale_value(c(NA, 1), scl))
  expect_error(bounded_scale_value(c(NA, NA), scl))
  expect_error(bounded_scale_value("a", scl))
  expect_error(bounded_scale_value(list(1.1), scl))
  expect_error(bounded_scale_value(distribution(c(1.1, 2.2)), scl))

  expect_equal(bounded_scale_value(NULL, scl), NA)
  expect_equal(bounded_scale_value(c(), scl), NA)
  expect_equal(bounded_scale_value(list(), scl), NA)
  expect_equal(bounded_scale_value(character(), scl), NA)
  expect_true(is.na(bounded_scale_value(NA, scl)))
  expect_true(is.na(bounded_scale_value(c(NA), scl)))
  expect_equal(bounded_scale_value(15.5, scl), 15.5)
  expect_equal(bounded_scale_value(distribution(15.5), scl), 15.5)
})

test_that("bounded_scale_value() works for discrete scales", {
  scl <- DexiDiscreteScale(values = c("low", "med", "high"))

  expect_error(bounded_scale_value(1, NULL))
  expect_error(bounded_scale_value(1, 2))
  expect_error(bounded_scale_value(scl, scl))
  expect_error(bounded_scale_value(c(NA, 1), scl))
  expect_error(bounded_scale_value(c(NA, NA), scl))
  expect_error(bounded_scale_value("lo", scl))
  expect_error(bounded_scale_value(c("a", "high"), scl))
  expect_error(bounded_scale_value(list("a", "high"), scl))

  expect_equal(bounded_scale_value(NULL, scl), NA)
  expect_equal(bounded_scale_value(c(), scl), NA)
  expect_equal(bounded_scale_value(list(), scl), NA)
  expect_equal(bounded_scale_value(NA, scl), NA)
  expect_equal(bounded_scale_value("*", scl), 1:3)
  expect_equal(bounded_scale_value("Undefined", scl), 1:3)
  expect_equal(bounded_scale_value(2, scl), 2)
  expect_equal(bounded_scale_value(c(-1, 2, 4), scl), c(2))
  expect_equal(bounded_scale_value(distribution(c(-1, 2, 4)), scl), distribution(c(-1, 2, 4)))
  expect_equal(bounded_scale_value(c(-1, 2.2, 4), scl), distribution(c(-1, 2.2, 4)))
  expect_equal(bounded_scale_value("high", scl), 3)
  expect_equal(bounded_scale_value(c("low", "high"), scl), c(1,3))
  v <- c(0.5, 0.4)
  names(v) <- c("low", "a")
  expect_error(bounded_scale_value(v, scl))
  names(v) <- c("low", "high")
  expect_equal(bounded_scale_value(v, scl), distribution(c(0.5, 0, 0.4)))

  expect_error(bounded_scale_value(list(a = 1.1, low = 2.2), scl))
  expect_equal(bounded_scale_value(list(high = 1.1, low = 2.2), scl), distribution(c(2.2, 0, 1.1)))

  expect_equal(bounded_scale_value(c(0.1, 0.2, 0.3, 0.4, 0.5), scl), distribution(c(0.1, 0.2, 0.3)))
  expect_equal(bounded_scale_value(distribution(c(0.1, 0.2, 0.3, 0.4, 0.5)), scl), distribution(c(0.1, 0.2, 0.3)))
})

test_that("value_text() works for continuous scales", {
  scl <- DexiContinuousScale()

  expect_equal(value_text(NULL, scl), "NULL")
  expect_equal(value_text(c(), scl), "NULL")
  expect_equal(value_text("", scl), "")
  expect_equal(value_text("something", scl), "something")
  expect_equal(value_text(character(), scl), "")
  expect_equal(value_text(list(), scl), "")
  expect_equal(value_text(data.frame(), scl), "")
  expect_equal(value_text(integer(), scl), "")
  expect_equal(value_text(c(TRUE, FALSE), scl), "TRUE FALSE")
  expect_equal(value_text(c(1, NA, 2), scl), "1 NA 2")
  expect_equal(value_text(c(1, "2"), scl), "1 2")
  expect_equal(value_text(c(1, 2, 3), scl), "1 2 3")
  expect_equal(value_text(5, scl), "5")
  expect_equal(value_text(-5, scl), "-5")
  expect_equal(value_text(1.1, scl), "1.1")
})

test_that("value_text() works for discrete scales", {
  scl <- DexiDiscreteScale(values = c("low", "med", "high"))

  expect_equal(value_text(NULL, scl), "NULL")
  expect_equal(value_text(c(), scl), "NULL")
  expect_equal(value_text("", scl), "")
  expect_equal(value_text("something", scl), "something")
  expect_equal(value_text(character(), scl), "")
  expect_equal(value_text(list(), scl), "")
  expect_equal(value_text(data.frame(), scl), "")
  expect_equal(value_text(integer(), scl), "")
  expect_equal(value_text(c(TRUE, FALSE), scl), "TRUE FALSE")
  expect_equal(value_text(c(1, NA, 2), scl), "1 NA 2")

  expect_equal(value_text(c(1, "2"), scl), "1 2")
  expect_equal(value_text(c(1, "2", 3), scl), "1 2 3")
  expect_equal(value_text(1:3, scl), "*")
  expect_equal(value_text(c(1, 3, 2), scl), "*")

  expect_equal(value_text(1, scl), "low")
  expect_equal(value_text(2:3, scl), "med, high")
  expect_equal(value_text(1:2, scl), "low, med")
  expect_equal(value_text(c(1, 3), scl), "low, high")

  expect_equal(value_text(5, scl), "NA")
  expect_equal(value_text(-5, scl), "NA")
  expect_equal(value_text(1.1, scl), "NA")
  expect_equal(value_text(-1:5, scl), "NA, NA, low, med, high, NA, NA")

  expect_error(value_text(distribution(), scl), "")
  expect_equal(value_text(distribution(0.5), scl), "'low'=0.5")
  expect_equal(value_text(distribution(0.5, 0, 0.1), scl), "'low'=0.5, 'high'=0.1")
  expect_equal(value_text(distribution(0, 0.5, 0.1), scl), "'med'=0.5, 'high'=0.1")
  expect_equal(value_text(distribution(0, 0.5, 0.1, 0.3), scl), "'med'=0.5, 'high'=0.1, 'NA'=0.3")

  v <- distribution(0.5, 0.4)
  names(v) <- c("low", "high")
  expect_equal(value_text(v, scl), "'low'=0.5, 'high'=0.4")
  names(v) <- c("low", "x")
  expect_equal(value_text(v, scl), "'low'=0.5, 'x'=0.4")

  expect_equal(value_text(list(low = 0.1, high = 0.5), scl), "'low'=0.1, 'high'=0.5")
  expect_equal(value_text(list(low = 0.1, x = 0.5), scl), "'low'=0.1, 'x'=0.5")
})

test_that("export_dexi_value() works", {
  expect_equal(export_dexi_value(NULL), "<undefined>")
  expect_equal(export_dexi_value(NA), "<undefined>")
  expect_equal(export_dexi_value(1), "1")
  expect_equal(export_dexi_value(3.2), "3.2")
  expect_equal(export_dexi_value(c(1, 3, 5)), "1;3;5")
  expect_equal(export_dexi_value(distribution(0.1, 0.9)), "1/0.1;2/0.9")
  expect_equal(export_dexi_value(distribution(0, 0.1, 0, 0.9, 0)), "2/0.1;4/0.9")
})


test_that("norm_max() works", {
  expect_error(norm_max())
  expect_equal(norm_max(1), 1)
  expect_equal(norm_max(0.5), 1)
  expect_equal(norm_max(0.5, max = 2), 2)
  expect_equal(norm_max(c(0.5, 1), max = 2), c(1, 2))
  expect_equal(norm_max(c(0.5, 1), max = 1), c(0.5, 1))
  expect_equal(norm_max(c(0.5, 1), max = 0.5), c(0.25, 0.5))
  expect_equal(norm_max(c(0.5, 1), max = 0.0), c(0, 0))
  expect_equal(norm_max(c(0.5, 1), max = -1), c(-0.5, -1.0))
  expect_equal(norm_max(c(0, 0), max = -1), c(0, 0))
  expect_equal(norm_max(c(0, 0), max = 1), c(0, 0))
})

test_that("norm_sum() works", {
  expect_error(norm_sum())
  expect_equal(norm_sum(1), 1)
  expect_equal(norm_sum(0.5), 1)
  expect_equal(norm_sum(0.5, sum = 2), 2)
  expect_equal(norm_sum(c(0.5, 1), sum = 2), c(2/3, 4/3))
  expect_equal(norm_sum(c(0.5, 1), sum = 1), c(1/3, 2/3))
  expect_equal(norm_sum(c(0.5, 1), sum = 0.5), c(1/6, 2/6))
  expect_equal(norm_sum(c(0.5, 1), sum = 0.0), c(0, 0))
  expect_equal(norm_sum(c(0.5, 1), sum = -1), c(-1/3, -2/3))
  expect_equal(norm_sum(c(0, 0), sum = -1), c(0, 0))
  expect_equal(norm_sum(c(0, 0), sum = 1), c(0, 0))
})

test_that("value_qualities() works for continuous scales", {
  scl <- DexiContinuousScale(low_point = -1, high_point = 1)

  expect_equal(value_qualities(NULL, scl), NA)
  expect_equal(value_qualities(NA, scl), NA)
  expect_equal(value_qualities(-2, scl), "bad")
  expect_equal(value_qualities(-1, scl), "none")
  expect_equal(value_qualities(-0.9, scl), "none")
  expect_equal(value_qualities( 0, scl), "none")
  expect_equal(value_qualities(+0.9, scl), "none")
  expect_equal(value_qualities(+1, scl), "none")
  expect_equal(value_qualities(+2, scl), "good")
  expect_equal(value_qualities(c(-2, -1, -0.9, 0, +0.9, +1, +2), scl), c("bad", "none", "none", "none", "none", "none", "good"))
})

test_that("value_qualities() works for discrete scales", {
  scl <- DexiDiscreteScale(values = c("low", "med", "high"))

  expect_equal(value_qualities(NULL, scl), NA)
  expect_equal(value_qualities(NA, scl), NA)
  expect_equal(value_qualities(-2, scl), NA)
  expect_equal(value_qualities(1, scl), "bad")
  expect_equal(value_qualities(2, scl), "none")
  expect_equal(value_qualities(3, scl), "good")

  expect_equal(value_qualities(1:3, scl), c("bad", "none", "good"))
  expect_equal(value_qualities(c(1,3), scl), c("bad", "good"))
  expect_equal(value_qualities(distribution(1, 0, 0.5), scl), c("bad", "good"))

  expect_equal(value_qualities(c(1,3,5), scl), c("bad", "good", NA))
  expect_equal(value_qualities(c(1,3,2.2), scl), c("bad", "good", "none"))
})

test_that("compare_values() works", {
  expect_equal(compare_values(c(1,2), c(1,2))      , 0)
  expect_equal(compare_values(c(1,2), c(1,3))      , NA)
  expect_equal(compare_values(c(1,2), c(3,4))      , -1)
  expect_equal(compare_values(c(1,2), c(2,4))      , NA)
  expect_equal(compare_values(c(1,2), c(2.1,4))    , -1)
  expect_equal(compare_values(c(1,2.05), c(2.1,4)) , -1)
  expect_equal(compare_values(c(3,4), c(3,4))      , 0)
  expect_equal(compare_values(c(5,5), c(3,4))      , +1)
  expect_equal(compare_values(c(5,5), 2)           , +1)
  expect_equal(compare_values(c(5,2), 2)           , NA)
  expect_equal(compare_values(c(5,3), 2)           , +1)
  expect_equal(compare_values(distribution(5,3), 2), NA)
  expect_equal(compare_values(distribution(5,3), 5), -1)
  expect_equal(compare_values("A", "B"),            NA)
  expect_equal(compare_values("A", 3),              NA)
  expect_equal(compare_values(list(5,3), list(5,3)),NA)
})

test_that("compare_values_by_preference() works", {
  expect_equal(compare_values_by_preference(1, 1, "none")      , 0)
  expect_equal(compare_values_by_preference(1, 2, "none")      , NA)
  expect_equal(compare_values_by_preference(3, 2, "none")      , NA)
  expect_equal(compare_values_by_preference(1, 1, "ascending") , 0)
  expect_equal(compare_values_by_preference(1, 2, "ascending") , -1)
  expect_equal(compare_values_by_preference(3, 2, "ascending") , +1)
  expect_equal(compare_values_by_preference(1, 1, "descending"), 0)
  expect_equal(compare_values_by_preference(1, 2, "descending"), +1)
  expect_equal(compare_values_by_preference(3, 2, "descending"), -1)
})

test_that("compare_values_on_scale() works", {
  expect_equal(compare_values_on_scale(1, 2, NULL)                      , NA)
  expect_equal(compare_values_on_scale(2, 1, "")                        , NA)
  expect_equal(compare_values_on_scale(1, 2, NULL, force_compare = TRUE), -1)
  expect_equal(compare_values_on_scale(2, 1, "", force_compare = TRUE)  , +1)

  scl <- DexiDiscreteScale(values = c("a", "b", "c"))
  expect_equal(compare_values_on_scale(1, 1, scl)            , 0)
  expect_equal(compare_values_on_scale(1, 2, scl)            , -1)
  expect_equal(compare_values_on_scale(3, 2, scl)            , +1)
  expect_equal(compare_values_on_scale(c(1, 2), c(1, 2), scl), 0)
  expect_equal(compare_values_on_scale(c(1, 2), c(2, 3), scl), NA)

  scl <- DexiDiscreteScale(order = "descending", values = c("a", "b", "c"))
  expect_equal(compare_values_on_scale(1, 1, scl)            , 0)
  expect_equal(compare_values_on_scale(1, 2, scl)            , +1)
  expect_equal(compare_values_on_scale(3, 2, scl)            , -1)
  expect_equal(compare_values_on_scale(c(1, 2), c(1, 2), scl), 0)
  expect_equal(compare_values_on_scale(c(1, 2), c(2, 3), scl), NA)
  scl <- DexiDiscreteScale(order = "none", values = c("a", "b", "c"))
  expect_equal(compare_values_on_scale(1, 1, scl)            , 0)
  expect_equal(compare_values_on_scale(1, 2, scl)            , NA)
  expect_equal(compare_values_on_scale(3, 2, scl)            , NA)
  expect_equal(compare_values_on_scale(c(1, 2), c(1, 2), scl), 0)
  expect_equal(compare_values_on_scale(c(1, 2), c(2, 3), scl), NA)

})

test_that("expand_value_to_points() works", {
  scl <- DexiDiscreteScale(values = c("L", "M", "H"))

  exp <- expand_value_to_points(c(1, 3), scl)
  expect_equal(nrow(exp), 2)
  expect_equal(names(exp), c("points", "sizes", "colors"))
  expect_equal(exp[["points"]], c(1, 3))
  expect_equal(exp[["sizes"]], c(1, 1))
  expect_equal(exp[["colors"]], c("red", "green"))

  exp <- expand_value_to_points(distribution(0.1, 0, 0.9), scl)
  expect_equal(nrow(exp), 2)
  expect_equal(names(exp), c("points", "sizes", "colors"))
  expect_equal(exp[["points"]], c(1, 3))
  expect_equal(exp[["sizes"]], c(0.1, 0.9))
  expect_equal(exp[["colors"]], c("red", "green"))
})
