# documentation separate from implementation because roxygen can't handle adding methods to another package's R6 classes

#' Create Azure Cosmos DB account
#'
#' Method for the [AzureRMR::az_resource_group] class.
#'
#' @rdname create_cosmosdb_account
#' @name create_cosmosdb_account
#' @aliases create_cosmosdb_account
#' @section Usage:
#' ```
#' create_cosmosdb_account(
#'     name,
#'     location = self$location,
#'     interface = c("sql", "cassandra", "mongo", "table", "graph"),
#'     serverless = FALSE,
#'     free_tier = FALSE,
#'     properties = list(),
#'     ...
#' )
#' ```
#' @section Arguments:
#' - `name`: The name of the Cosmos DB account.
#' - `location`: The location/region in which to create the account. Defaults to the resource group's location.
#' - `interface`: The default API by which to access data in the account.
#' - `serverless`: Whether this account should use provisioned throughput or a serverless mode. In the latter, you are charged solely on the basis of the traffic generated by your database operations. Serverless mode is best suited for small-to-medium workloads with light and intermittent traffic that is hard to forecast; it is currently (January 2021) in preview.
#' - `free_tier`: Whether this account should be in the free tier, in which a certain amount of database operations are provided free of charge. You can have one free tier account per subscription.
#' - `properties`: Additional properties to set for the account.
#' - `wait`: Whether to wait until the Cosmos DB account provisioning is complete.
#' - `...`: Optional arguments to pass to `az_cosmosdb$new()`.
#' @section Details:
#' This method creates a new Azure Cosmos DB account in the given resource group. Azure Cosmos DB is a globally distributed multi-model database that supports the document, graph, and key-value data models.
#'
#' The ARM resource object provides methods for working in the management plane. For working in the data plane, AzureCosmosR provides a client framework that interfaces with the core (SQL) API. Other packages provide functionality for other APIs, such as AzureTableStor for table storage and mongolite for MongoDB.
#' @section Value:
#' An object of class `az_cosmosdb` representing the Cosmos DB account.
#' @seealso
#' [get_cosmosdb_account], [delete_cosmosdb_account]
#'
#' For the SQL API client framework: [cosmos_endpoint], [cosmos_database], [cosmos_container], [query_documents]
#'
#' For the table storage API: [AzureTableStor::table_endpoint]
#'
#' For the MongoDB API: [cosmos_mongo_endpoint], [mongolite::mongo]
NULL


#' Get Azure Cosmos DB account
#'
#' Method for the [AzureRMR::az_resource_group] class.
#'
#' @rdname get_cosmosdb_account
#' @name get_cosmosdb_account
#' @aliases get_cosmosdb_account list_cosmosdb_accounts
#' @section Usage:
#' ```
#' get_cosmosdb_account(name)
#' list_cosmosdb_accounts()
#' ```
#' @section Arguments:
#' - `name`: The name of the Cosmos DB account.
#' @section Details:
#' `get_cosmosdb_account` retrieves the details for an existing Azure Cosmos DB account. `list_cosmosdb_accounts` retrieves all the Cosmos DB accounts within the resource group.
#' @section Value:
#' For `get_cosmosdb_account`, an object of class `az_cosmosdb` representing the Cosmos DB account. For `list_cosmosdb_accounts`, a list of such objects.
#' @seealso
#' [create_cosmosdb_account], [delete_cosmosdb_account]
#'
#' For the SQL API client framework: [cosmos_endpoint], [cosmos_database], [cosmos_container], [query_documents]
#'
#' For the table storage API: [AzureTableStor::table_endpoint]
#'
#' For the MongoDB API: [cosmos_mongo_endpoint], [mongolite::mongo]
NULL


#' Delete Azure Cosmos DB account
#'
#' Method for the [AzureRMR::az_resource_group] class.
#'
#' @rdname delete_cosmosdb_account
#' @name delete_cosmosdb_account
#' @aliases delete_cosmosdb_account
#' @section Usage:
#' ```
#' delete_cosmosdb_account(name, confirm = TRUE, wait = FALSE)
#' ```
#' @section Arguments:
#' - `name`: The name of the Cosmos DB account.
#' - `confirm`: Whether to ask for confirmation before deleting.
#' - `wait`: Whether to wait until the deletion has completed before returning.
#' @section Details:
#' This method deletes an existing Azure Cosmos DB account.
#' @seealso
#' [create_cosmosdb_account], [get_cosmosdb_account]
#'
#' For the SQL API client framework: [cosmos_endpoint], [cosmos_database], [cosmos_container], [query_documents]
#'
#' For the table storage API: [AzureTableStor::table_endpoint]
#'
#' For the MongoDB API: [cosmos_mongo_endpoint], [mongolite::mongo]
NULL


add_methods <- function()
{
    az_resource_group$set("public", "create_cosmosdb_account", overwrite=TRUE,
    function(name, location=self$location,
             interface=c("sql", "cassandra", "mongo", "table", "graph"),
             serverless=FALSE, free_tier=FALSE,
             properties=list(), wait=TRUE, ...)
    {
        interface <- match.arg(interface)
        kind <- if(interface == "mongo") "MongoDB" else "GlobalDocumentDB"

        capabilities <- if(interface == "cassandra")
            list(list(name="EnableCassandra"))
        else if(interface == "mongo")
            list(
                list(name="EnableMongo"),
                list(name="DisableRateLimitingResponses")
            )
        else if(interface == "table")
            list(list(name="EnableTable"))
        else if(interface == "graph")
            list(list(name="EnableGremlin"))
        else list()

        if(serverless)
            capabilities <- c(capabilities, list(list(name="EnableServerless")))

        properties <- utils::modifyList(properties, list(
            databaseAccountOfferType="standard",
            enableFreeTier=free_tier,
            capabilities=capabilities,
            locations=list(
                list(
                    id=paste0(name, "-", location),
                    failoverPriority=0,
                    locationName=location
                )
            )
        ))

        AzureCosmosR::az_cosmosdb$new(self$token, self$subscription, self$name,
            type="Microsoft.documentDB/databaseAccounts", name=name, location=location,
            kind=kind, properties=properties, wait=wait, ...)
    })

    az_resource_group$set("public", "get_cosmosdb_account", overwrite=TRUE,
    function(name)
    {
        AzureCosmosR::az_cosmosdb$new(self$token, self$subscription, self$name,
            type="Microsoft.documentDB/databaseAccounts", name=name)
    })

    az_resource_group$set("public", "list_cosmosdb_accounts", overwrite=TRUE,
    function()
    {
        provider <- "Microsoft.documentDB"
        path <- "databaseAccounts"
        api_version <- az_subscription$
            new(self$token, self$subscription)$
            get_provider_api_version(provider, path)

        op <- file.path("resourceGroups", self$name, "providers", provider, path)

        cont <- call_azure_rm(self$token, self$subscription, op, api_version=api_version)
        lst <- lapply(cont$value,
            function(parms) AzureCosmosR::az_cosmosdb$new(self$token, self$subscription, deployed_properties=parms))

        # keep going until paging is complete
        while(!is_empty(cont$nextLink))
        {
            cont <- call_azure_url(self$token, cont$nextLink)
            lst <- lapply(cont$value,
                function(parms) AzureCosmosR::az_cosmosdb$new(self$token, self$subscription, deployed_properties=parms))
        }
        named_list(lst)
    })

    az_resource_group$set("public", "delete_cosmosdb_account", overwrite=TRUE,
    function(name, confirm=TRUE, wait=FALSE)
    {
        self$get_cosmosdb_account(name)$delete(confirm=confirm, wait=wait)
    })
}
