% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/custom_cache.R
\name{custom_cache}
\alias{custom_cache}
\title{Create a custom cache backend (cachem-like)}
\usage{
custom_cache(get, set, remove, info = NULL)
}
\arguments{
\item{get}{A function(key, missing = NULL) -> value. Required.
Should return the stored value, or the \code{missing} argument if the key is not present.
The \code{missing} parameter is mandatory because both \code{OAuthClient} and
\code{OAuthProvider} validators will pass it explicitly.}

\item{set}{A function(key, value) -> invisible(NULL). Required.
Should store the value under the given key}

\item{remove}{A function(key) -> logical or sentinel. Required.

For state stores, this enforces single-use eviction. If your backend performs
an atomic "get-and-delete" (e.g., SQL DELETE .. RETURNING), you may supply
a function which does nothing here but returns \code{TRUE}. (The login flow will always attempt to call
\verb{$remove()} after \verb{$get()} as a best-effort cleanup.)

Recommended contract for interoperability and strong replay protection:
\itemize{
\item Return \code{TRUE} when a key was actually deleted or if it already did not exist
\item Return \code{FALSE} when they key could not be deleted or when it is unknown if
they key was deleted
}

When the return value is not \code{TRUE}, 'shinyOAuth' will attempt to retrieve
the value from the state store to check if it may still be present; if that
fails (i.e., key is not present), it will treat the removal as succesful.
If it does find the key, it will produce an error indicating that removal
did not succeed.}

\item{info}{Function() -> list(max_age = seconds, ...). Optional

This may be provided to because TTL information from \verb{$info()} is used to
align browser cookie max age in \code{oauth_module_server()}}
}
\value{
An R6 object exposing cachem-like \verb{$get/$set/$remove/$info} methods
}
\description{
Builds a minimal cachem-like cache backend object that exposes cachem-compatible methods:
\verb{$get(key, missing)}, \verb{$set(key, value)}, \verb{$remove(key)}, and \verb{$info()}.

Use this helper when you want to plug a custom state store or JWKS cache
into 'shinyOAuth', when \code{\link[cachem:cache_mem]{cachem::cache_mem()}} or \code{\link[cachem:cache_disk]{cachem::cache_disk()}}
are not suitable. This may be useful specifically when you deploy
a Shiny app to a multi-process environment with non-sticky workers.
In such cases, you may want to use a shared external cache (e.g., database,
Redis, Memcached).

The resulting object can be used in both places where 'shinyOAuth' accepts a cache-like object:
\itemize{
\item OAuthClient@state_store (requires \verb{$get}, \verb{$set}, \verb{$remove}; optional \verb{$info})
\item OAuthProvider@jwks_cache (requires \verb{$get}, \verb{$set}; optional \verb{$remove}, \verb{$info})
}

The \verb{$info()} method is optional, but if provided and it returns a list with
\code{max_age} (seconds), shinyOAuth will align cookie/issued_at TTLs to that value.
}
\examples{
mem <- new.env(parent = emptyenv())

my_cache <- custom_cache(
  get = function(key, missing = NULL) {
    base::get0(key, envir = mem, ifnotfound = missing, inherits = FALSE)
  },

  set = function(key, value) {
    assign(key, value, envir = mem)
    invisible(NULL)
  },

  remove = function(key) {
    if (exists(key, envir = mem, inherits = FALSE)) {
      rm(list = key, envir = mem)
      return(TRUE) # signal successful deletion
    }
    return(TRUE) # key did not exist
  },

  info = function() list(max_age = 600)
)
}
