#' Convert a genomic position to the CDS position 
#'
#' @param genome.position A positive integer specifying the genomic position
#' @param tx.id The Ensmble ID of the transcript
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' 
#' @return A CDS position
#' 
#' @importFrom ensembldb cdsBy
#' @importFrom ensembldb genomeToTranscript
#' @importFrom ensembldb transcriptToCds
#' @importFrom GenomicRanges GRanges
#' @importFrom GenomicRanges findOverlaps
#' @importFrom IRanges IRanges
#' @importFrom BiocGenerics strand
#' @importFrom BiocGenerics start
#' @importFrom BiocGenerics end
#' @importFrom GenomeInfoDb seqnames
#' 
#' @examples
#' \dontrun{
#'     genomeToCds()
#' }
#'
#' @noRd
genomeToCds <- function(genome.position, tx.id, ensdb){

    cds.list <- ensembldb::cdsBy(ensdb, filter = ~ tx_id==tx.id)[[1]]
    cds.range <- range(cds.list)
    strand <- as.character(BiocGenerics::strand(cds.range))
    chr.id <- as.character(GenomeInfoDb::seqnames(cds.range))
    cds.start <- BiocGenerics::start(cds.range)
    cds.end <- BiocGenerics::end(cds.range)

    if( genome.position<cds.start | genome.position>cds.end ){
    	cds.position <- NA
    }else{
	    genome.gr <- GenomicRanges::GRanges(chr.id, IRanges::IRanges(genome.position, width=1), strand=strand)
	    ovlp <- GenomicRanges::findOverlaps(genome.gr, cds.list, type="within")
	    if( length(ovlp)==0 ){
	        cds.position <- NA
	    }else{
	        tx.list <- ensembldb::genomeToTranscript(genome.gr, ensdb)[[1]]
	        tx.anno <- tx.list[tx.id]
	        cds.anno <- ensembldb::transcriptToCds(tx.anno, ensdb)
	        cds.position <- BiocGenerics::start(cds.anno)
	    }
    }
    
    return(cds.position)
}

