#' @details This function performs a proportion test on ranked set samples. It uses the method introduced by Chen et al. (2006), Zamanzade and Mahdizadeh (2020), and Ahn. et al. (2022) . Provide `data` as a data frame with columns `rank` and `y`. The function calculates the test statistic, confidence intervals, and p-value based on the RSS data.
#' @title RSS proportion test
#' @name rss.prop.test
#' @description The rss.prop.test function performs the population proportion test on ranked set sample data, supporting both balanced and unbalanced RSS designs.
#'
#' @param data A numeric data frame of ranked set samples with columns `rank` for ranks and `y` for data values.
#' @param alpha A numeric value specifying the confidence level for the interval.
#' @param alternative A character string specifying the alternative hypothesis. Must be one of "two.sided" (default), "greater", or "less".
#' @param p0 A numeric value indicating the hypothesized proportion for the one-sample test.
#'
#' @return
#'   \item{RSS_prop}{The RSS proportion estimate.}
#'   \item{CI}{The confidence interval for the population proportion.}
#'   \item{pstat}{The test statistic for the proportion test.}
#'   \item{p.value}{The p-value for the test.}
#'
#' @references
#'
#' Chen, H., Stasny, E. A., & Wolfe, D. A. (2006). Unbalanced ranked set sampling for estimating a population proportion. Biometrics, 62(1), 150-158.
#'
#' Zamanzade, E., & Mahdizadeh, M. (2020). Using ranked set sampling with extreme ranks in estimating the population proportion. Statistical methods in medical research, 29(1), 165-177.
#'
#' Ahn, S., Wang, X., Wang, M., & Lim, J. (2022). On continuity correction for RSS-structured cluster randomized designs with binary outcomes. Metron, 80(3), 383-397.
#'
#'
#' @seealso
#' \code{\link{rss.prop.simulation}}: used for simulating Ranked Set Samples (RSS) for proportions, which can serve as input.
#'
#' \code{\link{rss.prop.sampling}}: used for sampling Ranked Set Samples (RSS) from a population data set for proportions, providing input data.
#'
#' @importFrom methods is
#' @examples
#' ## Unbalanced RSS with a set size 3 and different sample sizes of 12, 9, 6 for each stratum,
#' ## with a population proportion of 0.6.
#' rss.prop.data=rss.prop.simulation(H=3,nsamp=c(12,9,6),p=0.6)
#'
#' ## RSS proportion test
#' rss.prop.test(data=rss.prop.data, alpha=0.05, alternative="two.sided", p0=0.2)
#'
#' @export
rss.prop.test <- function(data, alpha=0.05, alternative="two.sided", p0)
{
  alternative.set=c("two.sided", "less", "greater")
  if(!alternative %in% alternative.set) stop("Invalid alternative selected. Please choose from 'two.sided', 'less', or 'greater'.")

  if( (alpha > 0) & (alpha < 1)){

    if(!all(c("rank", "y") %in% colnames(data))) {
      stop("The input data must contain 'rank' and 'y' variables.")
    }
    H = length(unique(data$rank))
    nsamp = table(data$rank)
    n=sum(nsamp)

    phat = mean(tapply(data$y,data$rank,mean))

    phat.h=rep(NA,H)
    for (h in 1:H){
      phat.h[h]=1-stats::pbinom(H-h+1,H,phat)+stats::dbinom(H-h+1,H,phat)
    }
    varphat = sum(phat.h*(1-phat.h)/nsamp)/H^2

    # CI
    CI.up = phat + stats::qnorm(1-alpha/2) * sqrt(varphat)
    CI.low = phat - stats::qnorm(1-alpha/2) * sqrt(varphat)

    teststat = (phat-p0)/sqrt(varphat)

    if(alternative == "two.sided"){
      pval = 2*(1-stats::pnorm(abs(teststat)))
    }else if(alternative == "less"){
      pval = stats::pnorm(teststat)
    }else if(alternative == "greater"){
      pval = 1-stats::pnorm(teststat)
    }

    result = list(RSS_prop = phat, CI = c(CI.low, CI.up),  pstat = teststat, p.value = pval)
    return(result)
  }else stop("alpha is out of bound.", call. = F)
}
