% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Ops-extract.R
\name{OpsExtract}
\alias{OpsExtract}
\alias{[.Container}
\alias{[[.Container}
\title{Extract Parts of a Container}
\usage{
\method{[}{Container}(x, i, ..., .default = NULL)

\method{[[}{Container}(x, i)
}
\arguments{
\item{x}{A \code{Container} from which to extract elements.}

\item{i, ...}{Indices specifying elements to extract. Indices may be numeric,
character, logical, \code{NULL}, or empty. Logical vectors are recycled as needed.
Negative integers drop by position. Negative character indices drop by name.
Range expressions such as \code{a:b}, \code{1:c}, or \code{d:2} are supported for
convenience and are resolved in the calling environment.}

\item{.default}{A value used to fill missing items when extracting. If given,
unknown names and out-of-bounds positive indices are kept and filled with
this value.}
}
\value{
For \code{[} a \code{Container}. For \code{[[} the extracted value or \code{NULL}.
}
\description{
Operators that extract parts of a \code{Container}. The behavior is similar to
base R lists and includes convenient extensions for interactive work.
}
\details{
The \code{[} operator selects one or more elements and returns a \code{Container}.
Order is preserved and duplicates are kept. Logical indices recycle to the
container length with a warning when lengths do not match. \code{NA} in logical
indices is treated as \code{FALSE} with a warning. Positive and negative numeric
indices cannot be mixed in a single call and will raise an error.
Out-of-bounds negative indices are ignored. Character indices match names.
Unknown names are ignored unless \code{.default} is supplied, in which case they
are kept and filled. Comma-separated indices and \code{list(...)} are accepted and
behave like a single combined index.

The \code{[[} operator selects a single element and returns the value or \code{NULL}
if the element is not present.
}
\section{Warning}{

Range expressions such as \code{x[a:b]} are intended for interactive use, where
the resulting indices can be easily inspected and corrected by the user.
They are convenient for quick exploration and data analysis, but \emph{not}
intended for programming, where explicit indices should be used instead
to avoid unexpected results.
}

\examples{
co <- container(a = 1, b = 2, c = 3, d = 4)

# Numeric
co[c(1, 4)]                          # [a = 1, d = 4]
co[1, 4]                             # same (comma-sugar)
co[1, 1]                             # duplicates kept -> [a = 1, a = 1]
co[0:5]                              # unknowns ignored -> [a = 1, b = 2, c = 3, d = 4]
co[5]                                # [] (unknown positive index)

# Negative numeric
co[-c(1:2)]                          # [c = 3, d = 4]
co[-1, -4]                           # [b = 2, c = 3]
try(co[-1, 3])                       # error: cannot mix positive & negative
co[-5]                               # out-of-bounds negatives ignored -> full container

# Character
co[c("a", "d")]                      # [a = 1, d = 4]
co["a", "d"]                         # same
co[letters[1:5]]                     # unknown names dropped -> [a = 1, b = 2, c = 3, d = 4]
co["x"]                              # []

# Negative character (drop by name)
co[-c("a", "d")]                     # [b = 2, c = 3]
co[-"a", -"d"]                       # [b = 2, c = 3]

# Logical
co[c(TRUE, FALSE, TRUE, FALSE)]      # [a = 1, c = 3]
co[TRUE, FALSE]                      # [a = 1, c = 3] (recycled)
co[c(TRUE, NA)]                      # [a = 1, c = 3] (NA -> FALSE, warning)

# Mixed numeric and character
co[list(1, "d")]                     # [a = 1, d = 4]
co[1, "d"]                           # same

# Alphanumeric ranges (NSE)
co[a:b]                              # [a = 1, b = 2]
co[a:b, d:c]                         # [a = 1, b = 2, d = 4, c = 3]
co[1:c]                              # [a = 1, b = 2, c = 3]
co[d:2]                              # [d = 4, c = 3, b = 2]
co[-(a:c)]                           # [d = 4]

# Default-filling of missing items
co[1:5, 0, .default = 0]             # [a = 1, b = 2, c = 3, d = 4, 0]
co["a", "b", "z", .default = 0]      # [a = 1, b = 2, z = 0]
co[1:2, "z", .default = 3:4]         # [a = 1, b = 2, z = (3L 4L)]


co = container(a = 1, b = 2)
co[[1]]
co[["a"]]
co[["x"]]
}
\seealso{
\code{\link{peek_at}} for lenient extraction with defaults,
\code{\link{at}} and \code{\link{at2}} for strict programmatic access,
and base \code{\link[base:Extract]{[}}, \code{\link[base:Extract]{[[}}, and
\code{\link[base:Extract]{$}} for general indexing semantics.
}
