% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cld.R
\name{make_cld}
\alias{make_cld}
\alias{make_cld.pairwise.htest}
\alias{make_cld.PMCMR}
\alias{make_cld.PostHocTest}
\alias{make_cld.DunnTest}
\alias{make_cld.formula}
\alias{make_cld.matrix}
\alias{make_cld.data.frame}
\alias{make_cld.pairwise_pval_df}
\title{Make a compact letter display (CLD) for pair-wise comparisons}
\usage{
make_cld(obj, ..., alpha = 0.05)

\method{make_cld}{pairwise.htest}(obj, ..., alpha = 0.05)

\method{make_cld}{PMCMR}(obj, ..., alpha = 0.05)

\method{make_cld}{PostHocTest}(obj, ..., alpha = 0.05)

\method{make_cld}{DunnTest}(obj, ..., alpha = 0.05)

\method{make_cld}{formula}(obj, ..., data = NULL, alpha = 0.05)

\method{make_cld}{matrix}(obj, ..., alpha = 0.05)

\method{make_cld}{data.frame}(
  obj,
  ...,
  alpha = 0.05,
  gr1_col = "group1",
  gr2_col = "group2",
  p_val_col = "p.adj",
  remove_space = FALSE
)

\method{make_cld}{pairwise_pval_df}(obj, ..., alpha = 0.05)
}
\arguments{
\item{obj}{Object with pair-wise comparisons (e.g., post-hoc test results).
Currently supported object classes:
\itemize{
\item \code{PMCMR} - from packages \strong{PMCMR} and \strong{PMCMRplus}
\item \code{pairwise.htest} - from base R (e.g., \code{pairwise.t.test}, \code{pairwise.wilcox.test})
\item \code{data.frame} - with comparison results from packages like \strong{rstatix} (e.g., \code{games_howell_test}, \code{tukey_hsd}).
Requires \code{gr1_col}, \code{gr2_col}, \code{p_val_col} specification.
\item \code{PostHocTest} - from package \strong{DescTools} (e.g., \code{ConoverTest}, \code{DunnettTest})
\item \code{DunnTest} - from package \strong{DescTools}
\item \code{matrix} - symmetric matrices of p-values
\item \code{formula} - interface for data frames
\item \code{pairwise_pval_df} - output from \code{\link[=pval_matrix_to_df]{pval_matrix_to_df()}}
}}

\item{...}{Further arguments passed to internal methods. These may include:
\itemize{
\item \code{reversed} - Logical. If \code{TRUE}, reverses the letter ordering (default: \code{FALSE})
\item \code{swap_compared_names} - Logical. If \code{TRUE}, swaps group order in comparisons (default: \code{FALSE})
\item \code{print_comp} - Logical. If \code{TRUE}, prints comparison names (default: \code{FALSE})
\item \code{sep} - Character. Custom separator for comparison strings (default: \code{"-"}). Use this
when your group names contain hyphens. For example, use \code{sep = ":"} or \code{sep = ";"}.
Only applies to single-variable formula method. Not needed for two-variable formula.
Additional cleaning options (default: \code{TRUE} for most methods):
\item \code{remove_space} - Removes spaces from comparison strings
\item \code{remove_equal} - Removes equal signs from comparison strings
\item \code{swap_colon} - Replaces colons with hyphens (use \code{FALSE} if using \code{:} as separator)
\item \code{swap_vs} - Replaces "vs" with hyphens (default: \code{FALSE})
}}

\item{alpha}{Numeric value between 0 and 1. The significance level (alpha) for determining
which comparisons are significantly different. Comparisons with p-values below this
threshold are considered significant. Default is 0.05.}

\item{data}{A data frame with p-values and names of comparisons. This argument
is required when \code{obj} is a formula. The data frame should contain at least
two columns: one for p-values and one for comparison labels. See examples for details.}

\item{gr1_col}{Character string. Name of the column in the data frame containing
the first group names in each pairwise comparison. Default is \code{"group1"}.
Only used for the data.frame method. The function will construct comparisons
in the format \code{gr2-gr1}.}

\item{gr2_col}{Character string. Name of the column in the data frame containing
the second group names in each pairwise comparison. Default is \code{"group2"}.
Only used for the data.frame method.}

\item{p_val_col}{Character string. Name of the column in the data frame containing
the p-values for each comparison. Default is \code{"p.adj"} (adjusted p-values).
Only used for the data.frame method. Can also be \code{"p_value"} or any other
column name containing numeric p-values.}

\item{remove_space}{Logical. If \code{TRUE}, removes spaces from comparison strings.
Default is \code{FALSE} for the data.frame method to preserve original formatting.
Set to \code{TRUE} if your group names contain spaces and you want compact comparisons.}

\item{formula}{An R model \code{\link[stats:formula]{stats::formula()}} with two possible formats:
\itemize{
\item \strong{Two-variable formula} (recommended): \code{p_value ~ group1 + group2} where
\code{group1} and \code{group2} are separate columns containing group names. This format
automatically handles hyphens in group names.
\item \strong{Single-variable formula}: \code{p_value ~ Comparison} where \code{Comparison} is a
column with pre-formatted comparison strings (e.g., "A-B", "A-C"). This format
has limitations with hyphenated group names.
Usually used in combination with \code{data}.
}}
}
\value{
A data frame of class \code{c("cld_object", "data.frame")} with three columns:
\itemize{
\item \code{group} - Character. The names of the groups being compared
\item \code{cld} - Character. The compact letter display for each group. Groups sharing
at least one letter are not significantly different from each other
\item \code{spaced_cld} - Character. A monospaced version of the CLD where spaces are
replaced with underscores, useful for maintaining alignment in output
}

The rows are ordered according to the group names. Groups with the same letter(s)
do not differ significantly at the specified alpha level.
}
\description{
Creates a compact letter display for results of pair-wise comparisons
(e.g., ANOVA post-hoc tests, Kruskal-Wallis post-hoc tests, and others).
Groups that share the same letter are not significantly different from each other.
This provides a visual summary of which groups differ significantly, commonly used
in agricultural, biological, and statistical reporting.
}
\section{Handling Group Names with Hyphens}{

The underlying multcompView package uses hyphens (\code{-}) as the default separator
between group names in comparison strings (e.g., "GroupA-GroupB"). This creates
a conflict when group names themselves contain hyphens (e.g., "Plant-based",
"Treatment-1").

\strong{Automatic Handling:}
Most methods (matrix, data.frame, pairwise.htest, pairwise_pval_df, and others)
automatically detect and handle hyphens in group names by:
\enumerate{
\item Temporarily replacing hyphens with alternative characters (underscore, en-dash, etc.)
\item Processing the comparisons
\item Restoring the original hyphens in the output
}

An informational message is shown when this occurs. To suppress it, use
\code{quiet_hyphen_warning = TRUE}.

\strong{Formula Method Limitation:}
The formula method (e.g., \code{make_cld(p_value ~ comparison, data = df)}) has
limited support for group names with hyphens because it receives pre-formatted
comparison strings where the separator hyphens cannot be reliably distinguished
from hyphens within group names.

\strong{Best Practice for Hyphenated Group Names:}
Use the data.frame method with \code{gr1_col} and \code{gr2_col} parameters. This method
handles hyphens automatically and seamlessly. For example:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Instead of: make_cld(p_value ~ comparison, data = df)
# Use: make_cld(df, gr1_col = "group1", gr2_col = "group2", p_val_col = "p_value")
}\if{html}{\out{</div>}}

\strong{Alternative Workarounds for Formula Method:}
\itemize{
\item Convert your data to matrix format (also handles hyphens automatically)
\item Replace hyphens in group names with underscores before creating comparisons
\item Use a different separator in comparison strings (e.g., " vs " with \code{swap_vs = TRUE})
}
}

\examples{
# Example 1: Using pairwise.htest (Wilcoxon test)
obj1 <- stats::pairwise.wilcox.test(chickwts$weight, chickwts$feed, exact = FALSE)
cld::make_cld(obj1)

# Example 2: Using pairwise.htest (t-test)
obj2 <- with(OrchardSprays, stats::pairwise.t.test(decrease, treatment))
cld::make_cld(obj2)

# Example 3: Using pairwise.htest (proportion test)
\donttest{
smokers <- c(83, 90, 129, 70)
patients <- c(86, 93, 136, 82)
obj3 <- stats::pairwise.prop.test(smokers, patients)
cld::make_cld(obj3)
}

# Example 4: Using PMCMR objects
\donttest{
obj4 <- PMCMRplus::kwAllPairsConoverTest(count ~ spray, data = InsectSprays)
cld::make_cld(obj4)
}

# Example 5: Using DescTools objects
\donttest{
obj5 <- DescTools::ConoverTest(weight ~ group, data = PlantGrowth)
cld::make_cld(obj5)
}

# Example 6: Using rstatix data frames (via data.frame method)
\donttest{
obj6 <- rstatix::games_howell_test(PlantGrowth, weight ~ group)
cld::make_cld(obj6, gr1_col = "group1", gr2_col = "group2", p_val_col = "p.adj")
}

# Example 7: Using formula interface
my_dataframe <- utils::read.table(
  text = c(
    'Comparison     p_value p.adjust
    "EE - GB = 0"        1 1.000000
    "EE - CY = 0" 0.001093 0.003279
    "GB - CY = 0" 0.005477 0.008216'
  ),
  header = TRUE
)
cld::make_cld(p.adjust ~ Comparison, data = my_dataframe)

# Example 8: Using a symmetric matrix of p-values
# Create matrix
m <- c(
  1.00, 0.22, 0.05, 0.00,
  0.22, 1.00, 0.17, 0.01,
  0.05, 0.17, 1.00, 0.22,
  0.00, 0.01, 0.22, 1.00
)
obj8 <- matrix(m, nrow = 4)
rownames(obj8) <- colnames(obj8) <- c("P", "O", "I", "U")
obj8

# Make CLD
cld::make_cld(obj8)

# Example 9: Using data.frame method with custom column names
my_data <- data.frame(
  group1 = c("A", "A", "B"),
  group2 = c("B", "C", "C"),
  p.adj = c(0.001, 0.045, 0.892)
)
cld::make_cld(my_data, gr1_col = "group1", gr2_col = "group2", p_val_col = "p.adj")

}
\references{
Piepho HP (2004). An algorithm for a letter-based representation of
all-pairwise comparisons. Journal of Computational and Graphical Statistics,
13(2), 456-466. \doi{10.1198/1061860043515}, available at
https://www.tandfonline.com/doi/abs/10.1198/1061860043515
}
\seealso{
\strong{Helper Functions:}
\itemize{
\item \code{\link[=pval_matrix_to_df]{pval_matrix_to_df()}} for converting p-value matrices to data frames
\item \code{\link[=as_cld]{as_cld()}} for converting other formats to cld_object
}

\strong{Output Methods:}
\itemize{
\item \code{\link[=print.cld_object]{print.cld_object()}} for displaying CLD results
\item \link{cld_methods} for coercion methods (as.data.frame, as.character, as_tibble)
}

\strong{Underlying Algorithm:}
\itemize{
\item \code{\link[multcompView:multcompLetters]{multcompView::multcompLetters()}} for the letter generation algorithm
}
}
\concept{cld_creation}
\concept{compact letter display functions}
\concept{main functions}
\concept{main_functions}
\keyword{htest}
