% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/big_pls_cox_gd.R
\name{big_pls_cox_gd}
\alias{big_pls_cox_gd}
\title{Gradient-Descent Solver for Cox Models on Big Matrices}
\usage{
big_pls_cox_gd(
  X,
  time,
  status,
  ncomp = NULL,
  max_iter = 500L,
  tol = 1e-06,
  learning_rate = 0.01,
  keepX = NULL
)
}
\arguments{
\item{X}{A \code{\link[bigmemory:big.matrix-class]{bigmemory::big.matrix}} containing
the design matrix (rows are observations).}

\item{time}{A numeric vector of follow-up times with length equal to the
number of rows of \code{X}.}

\item{status}{A numeric or integer vector of the same length as \code{time}
containing the event indicators (1 for an event, 0 for censoring).}

\item{ncomp}{An integer giving the number of components (columns) to use from
\code{X}. Defaults to \code{min(5, ncol(X))}.}

\item{max_iter}{Maximum number of gradient-descent iterations (default 500).}

\item{tol}{Convergence tolerance on the Euclidean distance between successive
coefficient vectors.}

\item{learning_rate}{Step size used for the gradient-descent updates.}

\item{keepX}{Optional integer vector describing the number of predictors to
retain per component (naive sparsity). A value of zero keeps all
predictors.}
}
\value{
A list with components:
\itemize{
\item \code{coefficients}: Estimated Cox regression coefficients on the latent scores.
\item \code{loglik}: Final partial log-likelihood value.
\item \code{iterations}: Number of gradient-descent iterations performed.
\item \code{converged}: Logical flag indicating whether convergence was achieved.
\item \code{scores}: Matrix of latent score vectors (one column per component).
\item \code{loadings}: Matrix of loading vectors associated with each component.
\item \code{weights}: Matrix of PLS weight vectors.
\item \code{center}: Column means used to centre the predictors.
\item \code{scale}: Column scales (standard deviations) used to standardise the predictors.
}
}
\description{
Fits a Cox proportional hazards regression model using a gradient-descent
optimizer implemented in C++. The function operates directly on a
\code{\link[bigmemory:big.matrix-class]{bigmemory::big.matrix}} object to avoid
materialising large design matrices in memory.
}
\examples{
\donttest{
library(bigmemory)
set.seed(1)
n <- 50
p <- 10
X <- bigmemory::as.big.matrix(matrix(rnorm(n * p), n, p))
time <- rexp(n, rate = 0.1)
status <- rbinom(n, 1, 0.7)
fit <- big_pls_cox_gd(X, time, status, ncomp = 3, max_iter = 200)
}
}
\references{
Maumy, M., Bertrand, F. (2023). PLS models and their extension for big data.
Joint Statistical Meetings (JSM 2023), Toronto, ON, Canada.

Maumy, M., Bertrand, F. (2023). bigPLS: Fitting and cross-validating
PLS-based Cox models to censored big data. BioC2023 — The Bioconductor
Annual Conference, Dana-Farber Cancer Institute, Boston, MA, USA.
Poster. https://doi.org/10.7490/f1000research.1119546.1

Bastien, P., Bertrand, F., Meyer, N., & Maumy-Bertrand, M. (2015).
Deviance residuals-based sparse PLS and sparse kernel PLS for censored
data. \emph{Bioinformatics}, 31(3), 397–404. \url{doi:10.1093/bioinformatics/btu660}

Bertrand, F., Bastien, P., Meyer, N., & Maumy-Bertrand, M. (2014). PLS
models for censored data. In \emph{Proceedings of UseR! 2014} (p. 152).
}
\seealso{
\code{\link[=big_pls_cox]{big_pls_cox()}}, \code{\link[=predict.big_pls_cox]{predict.big_pls_cox()}}, \code{\link[=select_ncomp]{select_ncomp()}},
\code{\link[=computeDR]{computeDR()}}.
}
