#' @title Stress Tolerance (TOL)
#' @description This function computes stress tolerance (TOL) using any traits like yield under stress and non-stress conditions. The lower values of TOL indicates greater tolerance. For more details see Rosielle and Hamblin (1981) <doi:10.2135/cropsci1981.0011183X002100060033x>.
#'
#' @param Gen Character vector of Genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @return A list of output columns:
#' \itemize{
#'  \item \code{Gen}: Character vector of genotype IDs.
#'  \item \code{TOL}: Estimated stress tolerance (TOL) values for respective genotypes.
#' }
#' \deqn{TOL = YN - YS}
#'
#' @references Rosielle, A.A. and Hamblin, J. (1981). <doi:10.2135/cropsci1981.0011183X002100060033x>.
#' @examples
#' TOL(Gen=c("G1","G2","G3"), YN=c(10,8,5), YS=c(7,5,3))
#'
#' @export
TOL <- function(Gen, YN, YS) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  out <- YN - YS
  return(data.frame(Gen = as.character(Gen), TOL = out, row.names = NULL))
}
