#' @title Mean Relative Performance (MRP)
#' @description
#' This function computes Mean Relative Performance (MRP) using any traits (like yield) under stress and non-stress conditions. The lower values of MRP indicates greater tolerance. For more details see Ramirez-Vallejo and Kelly (1998) <doi:10.1023/A:1018353200015>.
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @param YMN Optional numeric scalar: mean of \code{YN}. If NULL, computed.
#' @param YMS Optional numeric scalar: mean of \code{YS}. If NULL, computed.
#' @return A list with \code{YMN}, \code{YMS}, and \code{Result} (data frame with \code{Gen}, \code{MRP}).
#'
#' \deqn{MRP = \left(\frac{YS}{YMS}\right) + \left(\frac{YN}{YMN}\right)}
#'
#' @references Ramirez-Vallejo, P. and Kelly, J.D. (1998). <doi:10.1023/A:1018353200015>.
#' @examples
#' out = MRP(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5),
#' YS=c(7,5,3)
#' )
#' print(out)
#' @export
MRP <- function(Gen, YN, YS, YMN = NULL, YMS = NULL) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  if (is.null(YMN)) YMN <- mean(YN, na.rm = TRUE)
  if (is.null(YMS)) YMS <- mean(YS, na.rm = TRUE)
  p1 <- YS / YMS; p1[YMS == 0] <- NA_real_
  p2 <- YN / YMN; p2[YMN == 0] <- NA_real_
  out <- p1 + p2
  Result <- data.frame(Gen = as.character(Gen), MRP = out, row.names = NULL)
  return(list(YMN = YMN, YMS = YMS, Result = Result))
}
