% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GPpenalty-package.R
\name{GPpenalty-package}
\alias{GPpenalty-package}
\title{GPpenalty}
\description{
Implements maximum likelihood estimation for Gaussian processes, supporting both isotropic and anisotropic models with predictive capabilities.
Includes penalized likelihood estimation using decorrelated prediction error-based metrics that account for uncertainty and cross validation techniques for tuning parameter selection.
Designed specifically for small datasets.
}
\section{Functions}{

\itemize{
\item \code{\link{mle_gp}}: The function computes maximum likelihood estimates for the lengthscale, scale, mu, and nugget (g) parameters using \code{optim},
with options to fix or assume zero for certain parameters.
\item \code{\link{predict_gp}}: Computes the posterior mean and covariance matrix for a given set of
input locations based on a fitted model.
\item \code{\link{gp_cv}}: Performs cross-validation to select an optimal tuning parameter for penalized MLE of the lengthscale parameter in Gaussian processes.
\item \code{\link{mle_penalty}}: Computes penalized maximum likelihood estimates for the lengthscale parameter using \code{optim}.
\item \code{\link{score}}: Calculates a score value. Higher score values indicate better fits.
\item \code{\link{dpe}}: Calculates a decorrelated prediction error value. Lower dpe values indicate better fits.
\item \code{\link{kernel}}: Compute the squared exponential kernel defined as \eqn{k = \exp(-\theta (x - x')^2) + g} , where \eqn{\theta} is the lengthscale parameter and \eqn{g} is a jitter term.
Both isotropic and separable kernels are supported.
}
}

\examples{
\donttest{
#### define function ###
f_x <- function(x) {
return(sin(2*pi*x) + x^2)
}

### x and y ###
x <- runif(8, min=0, max=1)
y <- f_x(x)
x.test <- runif(100, min=0, max=1)
y.test <- f_x(x.test)

### no penalization ###
# fit
fit <- mle_gp(y, x)
# prediction
pred <- predict_gp(fit, x.test)


# obtain kernel function
cov_function <- kernel(x1=x, theta=fit$theta)


# evaluate the predictive performance with score
score_value <- score(y.test, pred$mup, pred$Sigmap)

### penalization ###
# leave-one-out cross validation
loocv.lambda <- gp_cv(y, x)
# fit
fit.loocv <- mle_penalty(loocv.lambda)
# prediction
pred.loocv <- predict_gp(fit.loocv, x.test)

# k-fold cross validation with the dpe metric
kfold.dpe <- gp_cv(y, x, k=4)
# fit
fit.kfold.dpe <- mle_penalty(kfold.dpe)
# prediction
pred.kfold.dpe <- predict_gp(fit.kfold.dpe, x.test)

# k-fold cross validation with the mse metric
kfold.mse <- gp_cv(y, x, k=4, metric="mse")
# fit
fit.kfold.mse <- mle_penalty(kfold.mse)
# prediction
pred.kfold.mse <- predict_gp(fit.kfold.mse, x.test)
}



}
