/*
 *  R : A Computer Language for Statistical Data Analysis
 *  Copyright (C) 2005-2021   The R Core Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, a copy is available at
 *  https://www.R-project.org/Licenses/
 */

/* Split from rlocale_data.h for R 4.1.0. */

/* R i18n fake locale functions. */
enum { MB_Default, MB_ja_JP, MB_ko_KR, MB_zh_SG, MB_zh_CN, MB_zh_HK,
       MB_zh_TW, MB_SIZE }; // this sets MB_SIZE to 7

struct interval_wcwidth {
    int first;
    int last;
    char mb[MB_SIZE];
};

/* Background info for the first column is in
   http://www.unicode.org/Public/UCD/latest/ucd/EastAsianWidth.txt
   http://www.unicode.org/reports/tr11/

   For the 2020-05 updating, see
   https://github.com/brodieG/char-width

   2021-01 updated to Unicode 13.0.0, extracting the wide and
   fullwidth chars by the R script

tab <- read.table("EastAsianWidth.txt", header=FALSE, sep=";")
names(tab) <- c("range", "class")
tab <- tab[startsWith(tab$class, "W") |  startsWith(tab$class, "F"),]
xx <- strsplit(tab$range, "..", fixed = TRUE)
tab$first <- as.integer(paste0('0x',  sapply(xx, '[', 1)))
tab$last <- as.integer(paste0('0x', sapply(xx, function(x) x[length(x)])))
unlink("wd"); prev <- end <- 0
for(i in seq_along(tab$first))
{
    if (prev > 0 && tab$first[i] == tab$last[i-1] + 1) end <- tab$last[i]
    else {
        if(prev > 0)
            cat(sprintf("    {0x%x,0x%x,{2,2,2,2,2,2,2}},\n", prev, end),
                file = "wd", append = TRUE)
        prev <- tab$first[i]; end <- tab$last[i]
    }
}
cat(sprintf("    {0x%x,0x%x,{2,2,2,2,2,2,2}},\n",
            prev, end), file = "wd", append = TRUE)

Unicode 14.0 is expected ca 2021-09.
*/

/* Anything not in this table nor the zero-width one is width one */

/* glibc treats the ARIB STD 24 speed limit numbers U+3248-324F
 * (ambiguous in Unicode)
 * and the I Ching hexagrams U+4DC0-4DFF as wide.
 * (normal in Unicode tables).
 */

static const struct interval_wcwidth table_wcwidth[] = {
    {0xa1,0xa1,{1,2,2,1,2,1,1}}, // 'East Asian' ambiguous
    {0xa2,0xa3,{1,2,1,1,1,2,1}}, // Na
    {0xa4,0xa4,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0xa5,0xa5,{1,1,1,1,1,2,1}}, // Na
    {0xa6,0xa6,{1,2,1,1,1,1,1}}, // Na
    {0xa7,0xa7,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0xa8,0xa8,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0xa9,0xa9,{1,2,1,1,1,1,1}},
    {0xaa,0xaa,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xac,0xac,{1,2,1,1,1,1,1}}, // Na
    {0xad,0xad,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0xae,0xae,{1,2,2,1,2,1,1}}, // EA ambiguous:
    {0xaf,0xaf,{1,2,1,1,1,1,2}}, // Na
    {0xb0,0xb1,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0xb2,0xb3,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0xb4,0xb4,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xb6,0xb6,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xb7,0xb7,{1,1,2,2,2,2,2}}, // EA ambiguous
    {0xb8,0xb8,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xb9,0xb9,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0xba,0xba,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xbc,0xbe,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0xbf,0xbf,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xc0,0xc5,{1,2,1,1,1,1,1}},
    {0xc6,0xc6,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xc7,0xcf,{1,2,1,1,1,1,1}},
    {0xd0,0xd0,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0xd1,0xd6,{1,2,1,1,1,1,1}},
    {0xd7,0xd7,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0xd8,0xd8,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xd9,0xdd,{1,2,1,1,1,1,1}},
    {0xde,0xdf,{1,2,2,1,2,1,1}}, // EA ambiguous
#ifdef					Win32
    {0xe0,0xe1,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0xe0,0xe1,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0xe2,0xe5,{1,2,1,1,1,1,1}},
    {0xe6,0xe6,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xe7,0xe7,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0xe8,0xea,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0xe8,0xea,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0xeb,0xeb,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0xec,0xed,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0xec,0xed,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0xee,0xef,{1,2,1,1,1,1,1}},
    {0xf0,0xf0,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xf1,0xf1,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0xf2,0xf3,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0xf2,0xf3,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0xf4,0xf6,{1,2,1,1,1,1,1}},
    {0xf7,0xf7,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0xf8,0xf8,{1,2,2,1,2,1,1}}, // EA ambiguous
#ifdef					Win32
    {0xf9,0xfa,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0xf9,0xfa,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0xfb,0xfb,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0xfc,0xfc,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0xfc,0xfc,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0xfd,0xfd,{1,2,1,1,1,1,1}},
    {0xfe,0xfe,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0xff,0x100,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x101,0x101,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x101,0x101,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x102,0x110,{1,2,1,1,1,1,1}},
    {0x111,0x111,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x112,0x112,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x113,0x113,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x113,0x113,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x116,0x11a,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x11b,0x11b,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x11b,0x11b,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x11c,0x122,{1,2,1,1,1,1,1}},
    {0x124,0x125,{1,2,1,1,1,1,1}},
    {0x126,0x127,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x128,0x12a,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x12b,0x12b,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x12b,0x12b,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x12e,0x130,{1,2,1,1,1,1,1}},
    {0x131,0x133,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x134,0x137,{1,2,1,1,1,1,1}},
    {0x138,0x138,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x139,0x13e,{1,2,1,1,1,1,1}},
    {0x13f,0x142,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x143,0x143,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x144,0x144,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x144,0x144,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x145,0x147,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x148,0x148,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x148,0x148,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x149,0x14b,{1,2,2,1,2,1,1}},
    {0x14c,0x14c,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x14d,0x14d,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x14d,0x14d,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x150,0x151,{1,2,1,1,1,1,1}},
    {0x152,0x153,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x154,0x165,{1,2,1,1,1,1,1}},
    {0x166,0x167,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x168,0x16a,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x16b,0x16b,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x16b,0x16b,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x16c,0x17e,{1,2,1,1,1,1,1}},
    {0x1cd,0x1cd,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1ce,0x1ce,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1ce,0x1ce,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1cf,0x1cf,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1d0,0x1d0,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1d0,0x1d0,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1d1,0x1d1,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1d2,0x1d2,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1d2,0x1d2,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1d3,0x1d3,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1d4,0x1d4,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1d4,0x1d4,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1d5,0x1d5,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1d6,0x1d6,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1d6,0x1d6,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1d7,0x1d7,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1d8,0x1d8,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1d8,0x1d8,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1d9,0x1d9,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1da,0x1da,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1da,0x1da,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1db,0x1db,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x1dc,0x1dc,{1,2,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x1dc,0x1dc,{1,2,1,2,2,1,1}},
#endif					/* Win32 */
    {0x1f5,0x1f5,{1,2,1,1,1,1,1}},
#ifdef					Win32
    {0x251,0x251,{1,1,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x251,0x251,{1,1,1,2,2,1,1}},
#endif					/* Win32 */
#ifdef					Win32
    {0x261,0x261,{1,1,1,2,1,1,1}}, // EA ambiguous
#else					/* Win32 */
    {0x261,0x261,{1,1,1,2,2,1,1}},
#endif					/* Win32 */
    {0x2c4,0x2c4,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x2c7,0x2c7,{1,2,2,2,2,1,2}}, // EA ambiguous
    {0x2c9,0x2cb,{1,1,1,2,2,1,2}}, // EA ambiguous
    {0x2cd,0x2cd,{1,1,1,1,2,1,2}}, // EA ambiguous
    {0x2d0,0x2d0,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2d8,0x2d8,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2d9,0x2d9,{1,2,2,2,2,1,2}},
    {0x2da,0x2db,{1,2,2,1,2,1,1}},
    {0x2dd,0x2dd,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2df,0x2df,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x384,0x386,{1,2,1,1,1,1,1}},
    {0x388,0x38a,{1,2,1,1,1,1,1}},
    {0x38c,0x38c,{1,2,1,1,1,1,1}},
    {0x38e,0x390,{1,2,1,1,1,1,1}},
    {0x391,0x3a1,{1,2,2,2,2,1,2}}, // EA ambiguous
    {0x3a3,0x3a9,{1,2,2,2,2,1,2}}, // EA ambiguous
    {0x3aa,0x3b0,{1,2,1,1,1,1,1}},
    {0x3b1,0x3c1,{1,2,2,2,2,1,2}}, // EA ambiguous
    {0x3c2,0x3c2,{1,2,1,1,1,1,1}},
    {0x3c3,0x3c9,{1,2,2,2,2,1,2}}, // EA ambiguous
    {0x3ca,0x3ce,{1,2,1,1,1,1,1}},
    {0x401,0x401,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x402,0x40c,{1,2,1,1,1,1,1}},
    {0x40e,0x40f,{1,2,1,1,1,1,1}},
    {0x410,0x44f,{1,2,2,2,2,1,1}}, // Greek and Cyrillic,  EA ambiguous
    {0x451,0x451,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x452,0x45c,{1,2,1,1,1,1,1}},
    {0x45e,0x45f,{1,2,1,1,1,1,1}},
    {0x1100,0x115f,{2,2,2,2,2,2,2}}, // Hangul
    {0x2010,0x2010,{1,2,1,2,2,1,1}}, // EA ambiguous
    {0x2013,0x2014,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x2015,0x2015,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x2016,0x2016,{1,2,1,2,2,1,2}}, // EA ambiguous
    {0x2018,0x2019,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x201c,0x201d,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2020,0x2021,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2022,0x2022,{1,1,1,1,2,2,1}}, // EA ambiguous
    {0x2024,0x2024,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x2025,0x2026,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2027,0x2027,{1,1,1,1,2,1,2}}, // EA ambiguous
    {0x2030,0x2030,{1,2,2,2,2,1,1}}, // permille
    {0x2032,0x2032,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2033,0x2033,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x2035,0x2035,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x203b,0x203b,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x203e,0x203e,{1,1,1,1,2,2,2}}, // EA ambiguous
    {0x2074,0x2074,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x207f,0x207f,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2081,0x2084,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x20ac,0x20ac,{1,1,2,1,2,1,2}}, // EA ambiguous
    {0x2103,0x2103,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2105,0x2105,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x2109,0x2109,{1,1,2,2,2,2,2}}, // EA ambiguous
    {0x2113,0x2113,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2116,0x2116,{1,2,2,2,2,2,1}}, // EA ambiguous
    {0x2121,0x2121,{1,1,2,2,2,2,1}}, // EA ambiguous
    {0x2122,0x2122,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2126,0x2126,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x212b,0x212b,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2153,0x2154,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x215b,0x215e,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2160,0x2169,{1,1,2,2,2,2,2}}, // EA ambiguous
    {0x216a,0x216b,{1,1,1,2,2,1,1}}, // EA ambiguous
    {0x2170,0x2179,{1,1,2,2,2,2,2}}, // EA ambiguous
    {0x2190,0x2193,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2194,0x2195,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2196,0x2199,{1,1,2,2,2,2,2}}, // EA ambiguous
    {0x21b8,0x21b9,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x21d2,0x21d2,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x21d4,0x21d4,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x21e7,0x21e7,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x2200,0x2200,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2202,0x2203,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2207,0x2207,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2208,0x2208,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x220b,0x220b,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x220f,0x220f,{1,1,2,2,2,1,1}}, // EA ambiguous
    {0x2211,0x2211,{1,1,2,2,2,1,1}}, // EA ambiguous
    {0x2212,0x2212,{1,2,1,1,1,1,1}}, // minus sign
    {0x2215,0x2215,{1,1,1,2,2,1,2}}, // EA ambiguous
    {0x221a,0x221a,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x221d,0x221d,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x221e,0x221e,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x221f,0x221f,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x2220,0x2220,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2223,0x2223,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x2225,0x2225,{1,1,2,2,2,2,2}}, // EA ambiguous
    {0x2227,0x2228,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x2229,0x222b,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x222c,0x222c,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x222e,0x222e,{1,1,2,2,2,2,2}}, // EA ambiguous
    {0x2234,0x2235,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2236,0x2237,{1,1,1,2,2,1,1}}, // EA ambiguous
    {0x223c,0x223c,{1,1,2,1,2,2,2}}, // EA ambiguous
    {0x223d,0x223d,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x2248,0x2248,{1,1,1,2,2,1,1}}, // EA ambiguous
    {0x224c,0x224c,{1,1,1,2,2,1,1}}, // EA ambiguous
    {0x2252,0x2252,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2260,0x2261,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2264,0x2265,{1,1,2,2,2,1,1}}, // EA ambiguous
    {0x2266,0x2267,{1,2,1,2,2,2,2}}, // EA ambiguous
    {0x226a,0x226b,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x226e,0x226f,{1,1,1,2,2,1,1}}, // EA ambiguous
    {0x2282,0x2283,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2286,0x2287,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x2295,0x2295,{1,1,1,2,2,1,2}}, // EA ambiguous
    {0x2299,0x2299,{1,1,2,2,2,1,2}}, // EA ambiguous
    {0x22a5,0x22a5,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x22bf,0x22bf,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x2312,0x2312,{1,2,2,2,2,1,1}}, // EA ambiguous
    {0x231a,0x231b,{2,2,2,2,2,2,2}},
    {0x2329,0x232a,{2,2,2,2,2,2,2}}, // arrows (at least ends)
    {0x23e9,0x23ec,{2,2,2,2,2,2,2}},
    {0x23f0,0x23f0,{2,2,2,2,2,2,2}},
    {0x23f3,0x23f3,{2,2,2,2,2,2,2}},
    {0x2400,0x241f,{1,1,1,1,1,1,2}}, // control pictures
    {0x2421,0x2421,{1,1,1,1,1,1,2}},
    {0x2460,0x2469,{1,1,2,2,2,2,2}}, // EA ambiguous to 24e9
    {0x246a,0x246e,{1,1,2,1,2,1,1}},
    {0x246f,0x2473,{1,1,1,1,2,1,1}},
    {0x2474,0x247d,{1,1,2,2,2,2,2}},
    {0x247e,0x2482,{1,1,2,2,2,1,1}},
    {0x2483,0x249b,{1,1,1,2,2,1,1}},
    {0x249c,0x24b5,{1,1,2,1,2,1,1}},
    {0x24b6,0x24cf,{1,1,1,1,2,1,1}},
    {0x24d0,0x24e9,{1,1,2,1,2,1,1}},
    {0x24eb,0x24fe,{1,1,1,1,2,1,1}}, // EA ambiguous to 254b
    {0x2500,0x2500,{1,2,2,2,2,2,2}},
    {0x2501,0x2501,{1,2,2,2,2,1,1}},
    {0x2502,0x2502,{1,2,2,2,2,2,2}},
    {0x2503,0x2503,{1,2,2,2,2,1,1}},
    {0x2504,0x250b,{1,1,1,2,2,1,1}},
    {0x250c,0x250c,{1,2,2,2,2,2,2}},
    {0x250d,0x250e,{1,1,2,2,2,1,1}},
    {0x250f,0x250f,{1,2,2,2,2,1,1}},
    {0x2510,0x2510,{1,2,2,2,2,2,2}},
    {0x2511,0x2512,{1,1,2,2,2,1,1}},
    {0x2513,0x2513,{1,2,2,2,2,1,1}},
    {0x2514,0x2514,{1,2,2,2,2,2,2}},
    {0x2515,0x2516,{1,1,2,2,2,1,1}},
    {0x2517,0x2517,{1,2,2,2,2,1,1}},
    {0x2518,0x2518,{1,2,2,2,2,2,2}},
    {0x2519,0x251a,{1,1,2,2,2,1,1}},
    {0x251b,0x251b,{1,2,2,2,2,1,1}},
    {0x251c,0x251c,{1,2,2,2,2,2,2}},
    {0x251d,0x251d,{1,2,2,2,2,1,1}},
    {0x251e,0x251f,{1,1,2,2,2,1,1}},
    {0x2520,0x2520,{1,2,2,2,2,1,1}},
    {0x2521,0x2522,{1,1,2,2,2,1,1}},
    {0x2523,0x2523,{1,2,2,2,2,1,1}},
    {0x2524,0x2524,{1,2,2,2,2,2,2}},
    {0x2525,0x2525,{1,2,2,2,2,1,1}},
    {0x2526,0x2527,{1,1,2,2,2,1,1}},
    {0x2528,0x2528,{1,2,2,2,2,1,1}},
    {0x2529,0x252a,{1,1,2,2,2,1,1}},
    {0x252b,0x252b,{1,2,2,2,2,1,1}},
    {0x252c,0x252c,{1,2,2,2,2,2,2}},
    {0x252d,0x252e,{1,1,2,2,2,1,1}},
    {0x252f,0x2530,{1,2,2,2,2,1,1}},
    {0x2531,0x2532,{1,1,2,2,2,1,1}},
    {0x2533,0x2533,{1,2,2,2,2,1,1}},
    {0x2534,0x2534,{1,2,2,2,2,2,2}},
    {0x2535,0x2536,{1,1,2,2,2,1,1}},
    {0x2537,0x2538,{1,2,2,2,2,1,1}},
    {0x2539,0x253a,{1,1,2,2,2,1,1}},
    {0x253b,0x253b,{1,2,2,2,2,1,1}},
    {0x253c,0x253c,{1,2,2,2,2,2,2}},
    {0x253d,0x253e,{1,1,2,2,2,1,1}},
    {0x253f,0x253f,{1,2,2,2,2,1,1}},
    {0x2540,0x2541,{1,1,2,2,2,1,1}},
    {0x2542,0x2542,{1,2,2,2,2,1,1}},
    {0x2543,0x254a,{1,1,2,2,2,1,1}},
    {0x254b,0x254b,{1,2,2,2,2,1,1}},  // end of EA ambiguous
    {0x2550,0x2573,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x2574,0x2574,{1,1,1,1,1,1,2}},
    {0x2580,0x2580,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x2581,0x258f,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x2592,0x2592,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2593,0x2593,{1,1,1,2,2,1,2}}, // EA ambiguous
    {0x2594,0x2595,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x25a0,0x25a1,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x25a3,0x25a9,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x25b2,0x25b3,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x25b6,0x25b7,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x25bc,0x25bd,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x25c0,0x25c1,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x25c6,0x25c7,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x25c8,0x25c8,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x25cb,0x25cb,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x25ce,0x25cf,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x25d0,0x25d1,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x25e2,0x25e5,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x25ef,0x25ef,{1,2,1,1,2,1,1}}, // EA ambiguous
    {0x25fd,0x25fe,{2,2,2,2,2,2,2}},
    {0x2605,0x2606,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2609,0x2609,{1,1,1,2,2,2,2}}, // EA ambiguous
    {0x260e,0x260f,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2614,0x2615,{2,2,2,2,2,2,2}},
    {0x261c,0x261c,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x261e,0x261e,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2640,0x2640,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2641,0x2641,{1,1,1,1,1,2,2}},
    {0x2642,0x2642,{1,2,2,2,2,2,2}}, // EA ambiguous
    {0x2648,0x2653,{2,2,2,2,2,2,2}},
    {0x2660,0x2661,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2663,0x2665,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x2667,0x2669,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x266a,0x266a,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x266c,0x266c,{1,1,2,1,2,1,1}}, // EA ambiguous
    {0x266d,0x266d,{1,2,2,1,2,1,1}}, // EA ambiguous
    {0x266f,0x266f,{1,2,1,1,2,1,1}}, // EA ambiguous
    {0x267f,0x267f,{2,2,2,2,2,2,2}},
    {0x2693,0x2693,{2,2,2,2,2,2,2}},
    {0x26a1,0x26a1,{2,2,2,2,2,2,2}},
    {0x26aa,0x26ab,{2,2,2,2,2,2,2}},
    {0x26bd,0x26be,{2,2,2,2,2,2,2}},
    {0x26c4,0x26c5,{2,2,2,2,2,2,2}},
    {0x26ce,0x26ce,{2,2,2,2,2,2,2}},
    {0x26d4,0x26d4,{2,2,2,2,2,2,2}},
    {0x26ea,0x26ea,{2,2,2,2,2,2,2}},
    {0x26f2,0x26f3,{2,2,2,2,2,2,2}},
    {0x26f5,0x26f5,{2,2,2,2,2,2,2}},
    {0x26fa,0x26fa,{2,2,2,2,2,2,2}},
    {0x26fd,0x26fd,{2,2,2,2,2,2,2}},
    {0x2705,0x2705,{2,2,2,2,2,2,2}},
    {0x270a,0x270b,{2,2,2,2,2,2,2}},
    {0x2728,0x2728,{2,2,2,2,2,2,2}},
    {0x273d,0x273d,{1,1,1,1,2,2,1}},
    {0x273e,0x274b,{1,1,1,1,1,1,1}}, // EA ambiguous
    {0x274c,0x274c,{2,2,2,2,2,2,2}},
    {0x274e,0x274e,{2,2,2,2,2,2,2}},
    {0x2753,0x2755,{2,2,2,2,2,2,2}},
    {0x2757,0x2757,{2,2,2,2,2,2,2}},
    {0x2776,0x277f,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x2795,0x2797,{2,2,2,2,2,2,2}},
    {0x27b0,0x27b0,{2,2,2,2,2,2,2}},
    {0x27bf,0x27bf,{2,2,2,2,2,2,2}},
    {0x2b1b,0x2b1c,{2,2,2,2,2,2,2}},
    {0x2b50,0x2b50,{2,2,2,2,2,2,2}},
    {0x2b55,0x2b55,{2,2,2,2,2,2,2}},
    {0x2e80,0x2e99,{2,2,2,2,2,2,2}},
    {0x2e9b,0x2ef3,{2,2,2,2,2,2,2}},
    {0x2f00,0x2fd5,{2,2,2,2,2,2,2}},
    {0x2ff0,0x2ffb,{2,2,2,2,2,2,2}},
    {0x3000,0x3000,{2,2,2,2,2,2,2}}, // IDEOGRAPHIC SPACE
    {0x3001,0x3029,{2,2,2,2,2,2,2}},
    {0x302e,0x303e,{2,2,2,2,2,2,2}},
    {0x3041,0x3096,{2,2,2,2,2,2,2}},
    {0x309b,0x30ff,{2,2,2,2,2,2,2}},
    {0x3105,0x312f,{2,2,2,2,2,2,2}},
    {0x3131,0x318e,{2,2,2,2,2,2,2}},
    {0x3190,0x31e3,{2,2,2,2,2,2,2}},
    {0x31f0,0x321e,{2,2,2,2,2,2,2}},
    {0x3220,0x3247,{2,2,2,2,2,2,2}},
    {0x3250,0x4dbf,{2,2,2,2,2,2,2}},
    {0x4e00,0xa48c,{2,2,2,2,2,2,2}}, // CJK UNIFIED IDEOGRAPH and Yi
    {0xa490,0xa4c6,{2,2,2,2,2,2,2}}, // Yi
    {0xa960,0xa97c,{2,2,2,2,2,2,2}}, // HANGUL
    {0xac00,0xd7a3,{2,2,2,2,2,2,2}}, // Hangul
    {0xe000,0xe002,{1,1,1,1,2,1,1}}, // EA ambiguous to f8ff
    {0xe003,0xe003,{1,1,1,1,2,2,1}},
    {0xe004,0xe017,{1,1,1,1,2,1,1}},
    {0xe018,0xe018,{1,1,1,1,2,2,1}},
    {0xe019,0xe01a,{1,1,1,1,2,1,1}},
    {0xe01b,0xe01b,{1,1,1,1,2,2,1}},
    {0xe01c,0xe02c,{1,1,1,1,2,1,1}},
    {0xe02d,0xe02d,{1,1,1,1,2,2,1}},
    {0xe02e,0xe037,{1,1,1,1,2,1,1}},
    {0xe038,0xe038,{1,1,1,1,2,2,1}},
    {0xe039,0xe04b,{1,1,1,1,2,1,1}},
    {0xe04c,0xe04c,{1,1,1,1,2,2,1}},
    {0xe04d,0xe059,{1,1,1,1,2,1,1}},
    {0xe05a,0xe05a,{1,1,1,1,2,2,1}},
    {0xe05b,0xe093,{1,1,1,1,2,1,1}},
    {0xe094,0xe094,{1,1,1,1,2,2,1}},
    {0xe095,0xe098,{1,1,1,1,2,1,1}},
    {0xe099,0xe09a,{1,1,1,1,2,2,1}},
    {0xe09b,0xe0c4,{1,1,1,1,2,1,1}},
    {0xe0c5,0xe0c5,{1,1,1,1,2,2,1}},
    {0xe0c6,0xe0cf,{1,1,1,1,2,1,1}},
    {0xe0d0,0xe0d0,{1,1,1,1,2,2,1}},
    {0xe0d1,0xe0f1,{1,1,1,1,2,1,1}},
    {0xe0f2,0xe0f2,{1,1,1,1,2,2,1}},
    {0xe0f3,0xe0f8,{1,1,1,1,2,1,1}},
    {0xe0f9,0xe0f9,{1,1,1,1,2,2,1}},
    {0xe0fa,0xe0fc,{1,1,1,1,2,1,1}},
    {0xe0fd,0xe0fd,{1,1,1,1,2,2,1}},
    {0xe0fe,0xe105,{1,1,1,1,2,1,1}},
    {0xe106,0xe106,{1,1,1,1,2,2,1}},
    {0xe107,0xe108,{1,1,1,1,2,1,1}},
    {0xe109,0xe10a,{1,1,1,1,2,2,1}},
    {0xe10b,0xe12f,{1,1,1,1,2,1,1}},
    {0xe130,0xe130,{1,1,1,1,2,2,1}},
    {0xe131,0xe14f,{1,1,1,1,2,1,1}},
    {0xe150,0xe150,{1,1,1,1,2,2,1}},
    {0xe151,0xe153,{1,1,1,1,2,1,1}},
    {0xe154,0xe154,{1,1,1,1,2,2,1}},
    {0xe155,0xe15c,{1,1,1,1,2,1,1}},
    {0xe15d,0xe15d,{1,1,1,1,2,2,1}},
    {0xe15e,0xe167,{1,1,1,1,2,1,1}},
    {0xe168,0xe168,{1,1,1,1,2,2,1}},
    {0xe169,0xe169,{1,1,1,1,2,1,1}},
    {0xe16a,0xe16a,{1,1,1,1,2,2,1}},
    {0xe16b,0xe170,{1,1,1,1,2,1,1}},
    {0xe171,0xe171,{1,1,1,1,2,2,1}},
    {0xe172,0xe174,{1,1,1,1,2,1,1}},
    {0xe175,0xe175,{1,1,1,1,2,2,1}},
    {0xe176,0xe18e,{1,1,1,1,2,1,1}},
    {0xe18f,0xe18f,{1,1,1,1,2,2,1}},
    {0xe190,0xe1ba,{1,1,1,1,2,1,1}},
    {0xe1bb,0xe1bb,{1,1,1,1,2,2,1}},
    {0xe1bc,0xe1cb,{1,1,1,1,2,1,1}},
    {0xe1cc,0xe1cd,{1,1,1,1,2,2,1}},
    {0xe1ce,0xe1d6,{1,1,1,1,2,1,1}},
    {0xe1d7,0xe1d7,{1,1,1,1,2,2,1}},
    {0xe1d8,0xe1ee,{1,1,1,1,2,1,1}},
    {0xe1ef,0xe1ef,{1,1,1,1,2,2,1}},
    {0xe1f0,0xe1f1,{1,1,1,1,2,1,1}},
    {0xe1f2,0xe1f2,{1,1,1,1,2,2,1}},
    {0xe1f3,0xe1f9,{1,1,1,1,2,1,1}},
    {0xe1fa,0xe1fa,{1,1,1,1,2,2,1}},
    {0xe1fb,0xe1fd,{1,1,1,1,2,1,1}},
    {0xe1fe,0xe1fe,{1,1,1,1,2,2,1}},
    {0xe1ff,0xe201,{1,1,1,1,2,1,1}},
    {0xe202,0xe202,{1,1,1,1,2,2,1}},
    {0xe203,0xe20a,{1,1,1,1,2,1,1}},
    {0xe20b,0xe20b,{1,1,1,1,2,2,1}},
    {0xe20c,0xe218,{1,1,1,1,2,1,1}},
    {0xe219,0xe219,{1,1,1,1,2,2,1}},
    {0xe21a,0xe21b,{1,1,1,1,2,1,1}},
    {0xe21c,0xe21c,{1,1,1,1,2,2,1}},
    {0xe21d,0xe226,{1,1,1,1,2,1,1}},
    {0xe227,0xe227,{1,1,1,1,2,2,1}},
    {0xe228,0xe244,{1,1,1,1,2,1,1}},
    {0xe245,0xe245,{1,1,1,1,2,2,1}},
    {0xe246,0xe249,{1,1,1,1,2,1,1}},
    {0xe24a,0xe24a,{1,1,1,1,2,2,1}},
    {0xe24b,0xe24d,{1,1,1,1,2,1,1}},
    {0xe24e,0xe24e,{1,1,1,1,2,2,1}},
    {0xe24f,0xe26a,{1,1,1,1,2,1,1}},
    {0xe26b,0xe26b,{1,1,1,1,2,2,1}},
    {0xe26c,0xe279,{1,1,1,1,2,1,1}},
    {0xe27a,0xe27a,{1,1,1,1,2,2,1}},
    {0xe27b,0xe281,{1,1,1,1,2,1,1}},
    {0xe282,0xe282,{1,1,1,1,2,2,1}},
    {0xe283,0xe28f,{1,1,1,1,2,1,1}},
    {0xe290,0xe290,{1,1,1,1,2,2,1}},
    {0xe291,0xe29a,{1,1,1,1,2,1,1}},
    {0xe29b,0xe29b,{1,1,1,1,2,2,1}},
    {0xe29c,0xe2a0,{1,1,1,1,2,1,1}},
    {0xe2a1,0xe2a1,{1,1,1,1,2,2,1}},
    {0xe2a2,0xe2b1,{1,1,1,1,2,1,1}},
    {0xe2b2,0xe2b2,{1,1,1,1,2,2,1}},
    {0xe2b3,0xe2b6,{1,1,1,1,2,1,1}},
    {0xe2b7,0xe2b7,{1,1,1,1,2,2,1}},
    {0xe2b8,0xe2c0,{1,1,1,1,2,1,1}},
    {0xe2c1,0xe2c1,{1,1,1,1,2,2,1}},
    {0xe2c2,0xe2c8,{1,1,1,1,2,1,1}},
    {0xe2c9,0xe2c9,{1,1,1,1,2,2,1}},
    {0xe2ca,0xe2d3,{1,1,1,1,2,1,1}},
    {0xe2d4,0xe2d4,{1,1,1,1,2,2,1}},
    {0xe2d5,0xe2e1,{1,1,1,1,2,1,1}},
    {0xe2e2,0xe2e2,{1,1,1,1,2,2,1}},
    {0xe2e3,0xe2ea,{1,1,1,1,2,1,1}},
    {0xe2eb,0xe2ec,{1,1,1,1,2,2,1}},
    {0xe2ed,0xe2ef,{1,1,1,1,2,1,1}},
    {0xe2f0,0xe2f1,{1,1,1,1,2,2,1}},
    {0xe2f2,0xe2ff,{1,1,1,1,2,1,1}},
    {0xe300,0xe300,{1,1,1,1,2,2,1}},
    {0xe301,0xe310,{1,1,1,1,2,1,1}},
    {0xe311,0xe311,{1,1,1,1,2,2,1}},
    {0xe312,0xe324,{1,1,1,1,2,1,1}},
    {0xe325,0xe325,{1,1,1,1,2,2,1}},
    {0xe326,0xe32f,{1,1,1,1,2,1,1}},
    {0xe330,0xe330,{1,1,1,1,2,2,1}},
    {0xe331,0xe344,{1,1,1,1,2,1,1}},
    {0xe345,0xe345,{1,1,1,1,2,2,1}},
    {0xe346,0xe361,{1,1,1,1,2,1,1}},
    {0xe362,0xe362,{1,1,1,1,2,2,1}},
    {0xe363,0xe380,{1,1,1,1,2,1,1}},
    {0xe381,0xe382,{1,1,1,1,2,2,1}},
    {0xe383,0xe385,{1,1,1,1,2,1,1}},
    {0xe386,0xe386,{1,1,1,1,2,2,1}},
    {0xe387,0xe388,{1,1,1,1,2,1,1}},
    {0xe389,0xe389,{1,1,1,1,2,2,1}},
    {0xe38a,0xe393,{1,1,1,1,2,1,1}},
    {0xe394,0xe394,{1,1,1,1,2,2,1}},
    {0xe395,0xe395,{1,1,1,1,2,1,1}},
    {0xe396,0xe396,{1,1,1,1,2,2,1}},
    {0xe397,0xe39d,{1,1,1,1,2,1,1}},
    {0xe39e,0xe39e,{1,1,1,1,2,2,1}},
    {0xe39f,0xe39f,{1,1,1,1,2,1,1}},
    {0xe3a0,0xe3a0,{1,1,1,1,2,2,1}},
    {0xe3a1,0xe3b3,{1,1,1,1,2,1,1}},
    {0xe3b4,0xe3b4,{1,1,1,1,2,2,1}},
    {0xe3b5,0xe3ba,{1,1,1,1,2,1,1}},
    {0xe3bb,0xe3bb,{1,1,1,1,2,2,1}},
    {0xe3bc,0xe3cc,{1,1,1,1,2,1,1}},
    {0xe3cd,0xe3cd,{1,1,1,1,2,2,1}},
    {0xe3ce,0xe3ce,{1,1,1,1,2,1,1}},
    {0xe3cf,0xe3cf,{1,1,1,1,2,2,1}},
    {0xe3d0,0xe3d4,{1,1,1,1,2,1,1}},
    {0xe3d5,0xe3d5,{1,1,1,1,2,2,1}},
    {0xe3d6,0xe3e0,{1,1,1,1,2,1,1}},
    {0xe3e1,0xe3e1,{1,1,1,1,2,2,1}},
    {0xe3e2,0xe3e6,{1,1,1,1,2,1,1}},
    {0xe3e7,0xe3e7,{1,1,1,1,2,2,1}},
    {0xe3e8,0xe3ef,{1,1,1,1,2,1,1}},
    {0xe3f0,0xe3f0,{1,1,1,1,2,2,1}},
    {0xe3f1,0xe3f1,{1,1,1,1,2,1,1}},
    {0xe3f2,0xe3f2,{1,1,1,1,2,2,1}},
    {0xe3f3,0xe3f3,{1,1,1,1,2,1,1}},
    {0xe3f4,0xe3f4,{1,1,1,1,2,2,1}},
    {0xe3f5,0xe425,{1,1,1,1,2,1,1}},
    {0xe426,0xe426,{1,1,1,1,2,2,1}},
    {0xe427,0xe435,{1,1,1,1,2,1,1}},
    {0xe436,0xe436,{1,1,1,1,2,2,1}},
    {0xe437,0xe447,{1,1,1,1,2,1,1}},
    {0xe448,0xe448,{1,1,1,1,2,2,1}},
    {0xe449,0xe44c,{1,1,1,1,2,1,1}},
    {0xe44d,0xe44d,{1,1,1,1,2,2,1}},
    {0xe44e,0xe44f,{1,1,1,1,2,1,1}},
    {0xe450,0xe450,{1,1,1,1,2,2,1}},
    {0xe451,0xe466,{1,1,1,1,2,1,1}},
    {0xe467,0xe469,{1,1,1,1,2,2,1}},
    {0xe46a,0xe473,{1,1,1,1,2,1,1}},
    {0xe474,0xe474,{1,1,1,1,2,2,1}},
    {0xe475,0xe489,{1,1,1,1,2,1,1}},
    {0xe48a,0xe48a,{1,1,1,1,2,2,1}},
    {0xe48b,0xe4a0,{1,1,1,1,2,1,1}},
    {0xe4a1,0xe4a1,{1,1,1,1,2,2,1}},
    {0xe4a2,0xe4c0,{1,1,1,1,2,1,1}},
    {0xe4c1,0xe4c1,{1,1,1,1,2,2,1}},
    {0xe4c2,0xe4c5,{1,1,1,1,2,1,1}},
    {0xe4c6,0xe4c6,{1,1,1,1,2,2,1}},
    {0xe4c7,0xe4dc,{1,1,1,1,2,1,1}},
    {0xe4dd,0xe4dd,{1,1,1,1,2,2,1}},
    {0xe4de,0xe51d,{1,1,1,1,2,1,1}},
    {0xe51e,0xe51e,{1,1,1,1,2,2,1}},
    {0xe51f,0xe523,{1,1,1,1,2,1,1}},
    {0xe524,0xe524,{1,1,1,1,2,2,1}},
    {0xe525,0xe527,{1,1,1,1,2,1,1}},
    {0xe528,0xe528,{1,1,1,1,2,2,1}},
    {0xe529,0xe52d,{1,1,1,1,2,1,1}},
    {0xe52e,0xe52f,{1,1,1,1,2,2,1}},
    {0xe530,0xe56c,{1,1,1,1,2,1,1}},
    {0xe56d,0xe56d,{1,1,1,1,2,2,1}},
    {0xe56e,0xe56f,{1,1,1,1,2,1,1}},
    {0xe570,0xe570,{1,1,1,1,2,2,1}},
    {0xe571,0xe579,{1,1,1,1,2,1,1}},
    {0xe57a,0xe57a,{1,1,1,1,2,2,1}},
    {0xe57b,0xe57c,{1,1,1,1,2,1,1}},
    {0xe57d,0xe57d,{1,1,1,1,2,2,1}},
    {0xe57e,0xe583,{1,1,1,1,2,1,1}},
    {0xe584,0xe584,{1,1,1,1,2,2,1}},
    {0xe585,0xe58a,{1,1,1,1,2,1,1}},
    {0xe58b,0xe58b,{1,1,1,1,2,2,1}},
    {0xe58c,0xe599,{1,1,1,1,2,1,1}},
    {0xe59a,0xe59a,{1,1,1,1,2,2,1}},
    {0xe59b,0xe59d,{1,1,1,1,2,1,1}},
    {0xe59e,0xe59e,{1,1,1,1,2,2,1}},
    {0xe59f,0xe59f,{1,1,1,1,2,1,1}},
    {0xe5a0,0xe5a0,{1,1,1,1,2,2,1}},
    {0xe5a1,0xe5a5,{1,1,1,1,2,1,1}},
    {0xe5a6,0xe5a6,{1,1,1,1,2,2,1}},
    {0xe5a7,0xe5a9,{1,1,1,1,2,1,1}},
    {0xe5aa,0xe5aa,{1,1,1,1,2,2,1}},
    {0xe5ab,0xe5c1,{1,1,1,1,2,1,1}},
    {0xe5c2,0xe5c2,{1,1,1,1,2,2,1}},
    {0xe5c3,0xe5c7,{1,1,1,1,2,1,1}},
    {0xe5c8,0xe5c8,{1,1,1,1,2,2,1}},
    {0xe5c9,0xe5d7,{1,1,1,1,2,1,1}},
    {0xe5d8,0xe5d8,{1,1,1,1,2,2,1}},
    {0xe5d9,0xe5f0,{1,1,1,1,2,1,1}},
    {0xe5f1,0xe5f1,{1,1,1,1,2,2,1}},
    {0xe5f2,0xe618,{1,1,1,1,2,1,1}},
    {0xe619,0xe619,{1,1,1,1,2,2,1}},
    {0xe61a,0xe61b,{1,1,1,1,2,1,1}},
    {0xe61c,0xe61c,{1,1,1,1,2,2,1}},
    {0xe61d,0xe645,{1,1,1,1,2,1,1}},
    {0xe646,0xe646,{1,1,1,1,2,2,1}},
    {0xe647,0xe653,{1,1,1,1,2,1,1}},
    {0xe654,0xe654,{1,1,1,1,2,2,1}},
    {0xe655,0xe65f,{1,1,1,1,2,1,1}},
    {0xe660,0xe660,{1,1,1,1,2,2,1}},
    {0xe661,0xe669,{1,1,1,1,2,1,1}},
    {0xe66a,0xe66a,{1,1,1,1,2,2,1}},
    {0xe66b,0xe67c,{1,1,1,1,2,1,1}},
    {0xe67d,0xe67d,{1,1,1,1,2,2,1}},
    {0xe67e,0xe67f,{1,1,1,1,2,1,1}},
    {0xe680,0xe680,{1,1,1,1,2,2,1}},
    {0xe681,0xe68d,{1,1,1,1,2,1,1}},
    {0xe68e,0xe68e,{1,1,1,1,2,2,1}},
    {0xe68f,0xe696,{1,1,1,1,2,1,1}},
    {0xe697,0xe697,{1,1,1,1,2,2,1}},
    {0xe698,0xe69d,{1,1,1,1,2,1,1}},
    {0xe69e,0xe69e,{1,1,1,1,2,2,1}},
    {0xe69f,0xe6bf,{1,1,1,1,2,1,1}},
    {0xe6c0,0xe6c0,{1,1,1,1,2,2,1}},
    {0xe6c1,0xe6c4,{1,1,1,1,2,1,1}},
    {0xe6c5,0xe6c5,{1,1,1,1,2,2,1}},
    {0xe6c6,0xe6dd,{1,1,1,1,2,1,1}},
    {0xe6de,0xe6df,{1,1,1,1,2,2,1}},
    {0xe6e0,0xe700,{1,1,1,1,2,1,1}},
    {0xe701,0xe701,{1,1,1,1,2,2,1}},
    {0xe702,0xe712,{1,1,1,1,2,1,1}},
    {0xe713,0xe713,{1,1,1,1,2,2,1}},
    {0xe714,0xe724,{1,1,1,1,2,1,1}},
    {0xe725,0xe725,{1,1,1,1,2,2,1}},
    {0xe726,0xe73c,{1,1,1,1,2,1,1}},
    {0xe73d,0xe73d,{1,1,1,1,2,2,1}},
    {0xe73e,0xe740,{1,1,1,1,2,1,1}},
    {0xe741,0xe741,{1,1,1,1,2,2,1}},
    {0xe742,0xe751,{1,1,1,1,2,1,1}},
    {0xe752,0xe752,{1,1,1,1,2,2,1}},
    {0xe753,0xe75c,{1,1,1,1,2,1,1}},
    {0xe75d,0xe75d,{1,1,1,1,2,2,1}},
    {0xe75e,0xe760,{1,1,1,1,2,1,1}},
    {0xe761,0xe761,{1,1,1,1,2,2,1}},
    {0xe762,0xe767,{1,1,1,1,2,1,1}},
    {0xe768,0xe768,{1,1,1,1,2,2,1}},
    {0xe769,0xe774,{1,1,1,1,2,1,1}},
    {0xe775,0xe775,{1,1,1,1,2,2,1}},
    {0xe776,0xe779,{1,1,1,1,2,1,1}},
    {0xe77a,0xe77a,{1,1,1,1,2,2,1}},
    {0xe77b,0xe77c,{1,1,1,1,2,1,1}},
    {0xe77d,0xe77d,{1,1,1,1,2,2,1}},
    {0xe77e,0xe78f,{1,1,1,1,2,1,1}},
    {0xe790,0xe790,{1,1,1,1,2,2,1}},
    {0xe791,0xe796,{1,1,1,1,2,1,1}},
    {0xe797,0xe797,{1,1,1,1,2,2,1}},
    {0xe798,0xe7a5,{1,1,1,1,2,1,1}},
    {0xe7a6,0xe7a6,{1,1,1,1,2,2,1}},
    {0xe7a7,0xe7b3,{1,1,1,1,2,1,1}},
    {0xe7b4,0xe7b4,{1,1,1,1,2,2,1}},
    {0xe7b5,0xe7c6,{1,1,1,1,2,1,1}},
#ifdef					Win32
    {0xe7c7,0xe7c7,{1,1,1,1,1,1,1}},
#else					/* Win32 */
    {0xe7c7,0xe7c7,{1,1,1,1,2,1,1}},
#endif					/* Win32 */
    {0xe7c8,0xe7d9,{1,1,1,1,2,1,1}},
    {0xe7da,0xe7da,{1,1,1,1,2,2,1}},
    {0xe7db,0xe7dc,{1,1,1,1,2,1,1}},
    {0xe7dd,0xe7dd,{1,1,1,1,2,2,1}},
    {0xe7de,0xe7e7,{1,1,1,1,2,1,1}},
    {0xe7e8,0xe7e9,{1,1,1,1,2,2,1}},
    {0xe7ea,0xe7ec,{1,1,1,1,2,1,1}},
    {0xe7ed,0xe7ed,{1,1,1,1,2,2,1}},
    {0xe7ee,0xe7f0,{1,1,1,1,2,1,1}},
    {0xe7f1,0xe7f1,{1,1,1,1,2,2,1}},
    {0xe7f2,0xe7fa,{1,1,1,1,2,1,1}},
    {0xe7fb,0xe7fc,{1,1,1,1,2,2,1}},
    {0xe7fd,0xe7ff,{1,1,1,1,2,1,1}},
    {0xe800,0xe800,{1,1,1,1,2,2,1}},
    {0xe801,0xe80b,{1,1,1,1,2,1,1}},
    {0xe80c,0xe80c,{1,1,1,1,2,2,1}},
    {0xe80d,0xe812,{1,1,1,1,2,1,1}},
    {0xe813,0xe813,{1,1,1,1,2,2,1}},
    {0xe814,0xe820,{1,1,1,1,2,1,1}},
    {0xe821,0xe821,{1,1,1,1,2,2,1}},
    {0xe822,0xe827,{1,1,1,1,2,1,1}},
    {0xe828,0xe828,{1,1,1,1,2,2,1}},
    {0xe829,0xe82c,{1,1,1,1,2,1,1}},
    {0xe82d,0xe82d,{1,1,1,1,2,2,1}},
    {0xe82e,0xe82e,{1,1,1,1,2,1,1}},
    {0xe82f,0xe82f,{1,1,1,1,2,2,1}},
    {0xe830,0xe84c,{1,1,1,1,2,1,1}},
    {0xe84d,0xe84d,{1,1,1,1,2,2,1}},
    {0xe84e,0xe854,{1,1,1,1,2,1,1}},
    {0xe855,0xe856,{1,1,1,1,2,2,1}},
    {0xe857,0xe85a,{1,1,1,1,2,1,1}},
    {0xe85b,0xe85b,{1,1,1,1,2,2,1}},
    {0xe85c,0xe85c,{1,1,1,1,2,1,1}},
    {0xe85d,0xe85d,{1,1,1,1,2,2,1}},
    {0xe85e,0xe864,{1,1,1,1,2,1,1}},
    {0xe865,0xe865,{1,1,1,1,2,2,1}},
    {0xe866,0xe886,{1,1,1,1,2,1,1}},
    {0xe887,0xe887,{1,1,1,1,2,2,1}},
    {0xe888,0xe88e,{1,1,1,1,2,1,1}},
    {0xe88f,0xe88f,{1,1,1,1,2,2,1}},
    {0xe890,0xe890,{1,1,1,1,2,1,1}},
    {0xe891,0xe891,{1,1,1,1,2,2,1}},
    {0xe892,0xe8a0,{1,1,1,1,2,1,1}},
    {0xe8a1,0xe8a1,{1,1,1,1,2,2,1}},
    {0xe8a2,0xe8a3,{1,1,1,1,2,1,1}},
    {0xe8a4,0xe8a4,{1,1,1,1,2,2,1}},
    {0xe8a5,0xe8c1,{1,1,1,1,2,1,1}},
    {0xe8c2,0xe8c3,{1,1,1,1,2,2,1}},
    {0xe8c4,0xe8cc,{1,1,1,1,2,1,1}},
    {0xe8cd,0xe8cd,{1,1,1,1,2,2,1}},
    {0xe8ce,0xe8d4,{1,1,1,1,2,1,1}},
    {0xe8d5,0xe8d6,{1,1,1,1,2,2,1}},
    {0xe8d7,0xe8d8,{1,1,1,1,2,1,1}},
    {0xe8d9,0xe8da,{1,1,1,1,2,2,1}},
    {0xe8db,0xe8e0,{1,1,1,1,2,1,1}},
    {0xe8e1,0xe8e1,{1,1,1,1,2,2,1}},
    {0xe8e2,0xe8e4,{1,1,1,1,2,1,1}},
    {0xe8e5,0xe8e5,{1,1,1,1,2,2,1}},
    {0xe8e6,0xe8fd,{1,1,1,1,2,1,1}},
    {0xe8fe,0xe8fe,{1,1,1,1,2,2,1}},
    {0xe8ff,0xe916,{1,1,1,1,2,1,1}},
    {0xe917,0xe918,{1,1,1,1,2,2,1}},
    {0xe919,0xe91e,{1,1,1,1,2,1,1}},
    {0xe91f,0xe91f,{1,1,1,1,2,2,1}},
    {0xe920,0xe924,{1,1,1,1,2,1,1}},
    {0xe925,0xe925,{1,1,1,1,2,2,1}},
    {0xe926,0xe92a,{1,1,1,1,2,1,1}},
    {0xe92b,0xe92b,{1,1,1,1,2,2,1}},
    {0xe92c,0xe933,{1,1,1,1,2,1,1}},
    {0xe934,0xe934,{1,1,1,1,2,2,1}},
    {0xe935,0xe94f,{1,1,1,1,2,1,1}},
    {0xe950,0xe950,{1,1,1,1,2,2,1}},
    {0xe951,0xe966,{1,1,1,1,2,1,1}},
    {0xe967,0xe967,{1,1,1,1,2,2,1}},
    {0xe968,0xe96d,{1,1,1,1,2,1,1}},
    {0xe96e,0xe96f,{1,1,1,1,2,2,1}},
    {0xe970,0xe972,{1,1,1,1,2,1,1}},
    {0xe973,0xe973,{1,1,1,1,2,2,1}},
    {0xe974,0xe976,{1,1,1,1,2,1,1}},
    {0xe977,0xe977,{1,1,1,1,2,2,1}},
    {0xe978,0xe979,{1,1,1,1,2,1,1}},
    {0xe97a,0xe97b,{1,1,1,1,2,2,1}},
    {0xe97c,0xe97c,{1,1,1,1,2,1,1}},
    {0xe97d,0xe97d,{1,1,1,1,2,2,1}},
    {0xe97e,0xe981,{1,1,1,1,2,1,1}},
    {0xe982,0xe982,{1,1,1,1,2,2,1}},
    {0xe983,0xe990,{1,1,1,1,2,1,1}},
    {0xe991,0xe991,{1,1,1,1,2,2,1}},
    {0xe992,0xe9ca,{1,1,1,1,2,1,1}},
    {0xe9cb,0xe9cb,{1,1,1,1,2,2,1}},
    {0xe9cc,0xeaa8,{1,1,1,1,2,1,1}},
    {0xeaa9,0xeaa9,{1,1,1,1,2,2,1}},
    {0xeaaa,0xeabe,{1,1,1,1,2,1,1}},
    {0xeabf,0xeac1,{1,1,1,1,2,2,1}},
    {0xeac2,0xeac4,{1,1,1,1,2,1,1}},
    {0xeac5,0xeac5,{1,1,1,1,2,2,1}},
    {0xeac6,0xeac8,{1,1,1,1,2,1,1}},
    {0xeac9,0xeacc,{1,1,1,1,2,2,1}},
    {0xeacd,0xeace,{1,1,1,1,2,1,1}},
    {0xeacf,0xead1,{1,1,1,1,2,2,1}},
    {0xead2,0xead2,{1,1,1,1,2,1,1}},
    {0xead3,0xead4,{1,1,1,1,2,2,1}},
    {0xead5,0xead5,{1,1,1,1,2,1,1}},
    {0xead6,0xead7,{1,1,1,1,2,2,1}},
    {0xead8,0xead8,{1,1,1,1,2,1,1}},
    {0xead9,0xeada,{1,1,1,1,2,2,1}},
    {0xeadb,0xeadb,{1,1,1,1,2,1,1}},
    {0xeadc,0xeadc,{1,1,1,1,2,2,1}},
    {0xeadd,0xeadd,{1,1,1,1,2,1,1}},
    {0xeade,0xeae3,{1,1,1,1,2,2,1}},
    {0xeae4,0xeae6,{1,1,1,1,2,1,1}},
    {0xeae7,0xeae7,{1,1,1,1,2,2,1}},
    {0xeae8,0xeae8,{1,1,1,1,2,1,1}},
    {0xeae9,0xeaea,{1,1,1,1,2,2,1}},
    {0xeaeb,0xeaeb,{1,1,1,1,2,1,1}},
    {0xeaec,0xeaec,{1,1,1,1,2,2,1}},
    {0xeaed,0xeaed,{1,1,1,1,2,1,1}},
    {0xeaee,0xeaee,{1,1,1,1,2,2,1}},
    {0xeaef,0xeaef,{1,1,1,1,2,1,1}},
    {0xeaf0,0xeaf1,{1,1,1,1,2,2,1}},
    {0xeaf2,0xeaf4,{1,1,1,1,2,1,1}},
    {0xeaf5,0xeaf6,{1,1,1,1,2,2,1}},
    {0xeaf7,0xeaf7,{1,1,1,1,2,1,1}},
    {0xeaf8,0xeafb,{1,1,1,1,2,2,1}},
    {0xeafc,0xeafe,{1,1,1,1,2,1,1}},
    {0xeaff,0xeb00,{1,1,1,1,2,2,1}},
    {0xeb01,0xeb01,{1,1,1,1,2,1,1}},
    {0xeb02,0xeb05,{1,1,1,1,2,2,1}},
    {0xeb06,0xeb07,{1,1,1,1,2,1,1}},
    {0xeb08,0xeb0f,{1,1,1,1,2,2,1}},
    {0xeb10,0xeb11,{1,1,1,1,2,1,1}},
    {0xeb12,0xeb12,{1,1,1,1,2,2,1}},
    {0xeb13,0xeb14,{1,1,1,1,2,1,1}},
    {0xeb15,0xeb15,{1,1,1,1,2,2,1}},
    {0xeb16,0xeb18,{1,1,1,1,2,1,1}},
    {0xeb19,0xeb1d,{1,1,1,1,2,2,1}},
    {0xeb1e,0xeb1e,{1,1,1,1,2,1,1}},
    {0xeb1f,0xeb20,{1,1,1,1,2,2,1}},
    {0xeb21,0xeb22,{1,1,1,1,2,1,1}},
    {0xeb23,0xeb23,{1,1,1,1,2,2,1}},
    {0xeb24,0xeb24,{1,1,1,1,2,1,1}},
    {0xeb25,0xeb25,{1,1,1,1,2,2,1}},
    {0xeb26,0xeb26,{1,1,1,1,2,1,1}},
    {0xeb27,0xeb27,{1,1,1,1,2,2,1}},
    {0xeb28,0xeb29,{1,1,1,1,2,1,1}},
    {0xeb2a,0xeb2a,{1,1,1,1,2,2,1}},
    {0xeb2b,0xeb2b,{1,1,1,1,2,1,1}},
    {0xeb2c,0xeb2e,{1,1,1,1,2,2,1}},
    {0xeb2f,0xeb2f,{1,1,1,1,2,1,1}},
    {0xeb30,0xeb39,{1,1,1,1,2,2,1}},
    {0xeb3a,0xeb3c,{1,1,1,1,2,1,1}},
    {0xeb3d,0xeb3f,{1,1,1,1,2,2,1}},
    {0xeb40,0xeb42,{1,1,1,1,2,1,1}},
    {0xeb43,0xeb44,{1,1,1,1,2,2,1}},
    {0xeb45,0xeb49,{1,1,1,1,2,1,1}},
    {0xeb4a,0xeb4a,{1,1,1,1,2,2,1}},
    {0xeb4b,0xeb4c,{1,1,1,1,2,1,1}},
    {0xeb4d,0xeb4e,{1,1,1,1,2,2,1}},
    {0xeb4f,0xeb52,{1,1,1,1,2,1,1}},
    {0xeb53,0xeb53,{1,1,1,1,2,2,1}},
    {0xeb54,0xeb54,{1,1,1,1,2,1,1}},
    {0xeb55,0xeb56,{1,1,1,1,2,2,1}},
    {0xeb57,0xeb57,{1,1,1,1,2,1,1}},
    {0xeb58,0xeb58,{1,1,1,1,2,2,1}},
    {0xeb59,0xeb59,{1,1,1,1,2,1,1}},
    {0xeb5a,0xeb5b,{1,1,1,1,2,2,1}},
    {0xeb5c,0xeb61,{1,1,1,1,2,1,1}},
    {0xeb62,0xeb63,{1,1,1,1,2,2,1}},
    {0xeb64,0xeb64,{1,1,1,1,2,1,1}},
    {0xeb65,0xeb65,{1,1,1,1,2,2,1}},
    {0xeb66,0xeb66,{1,1,1,1,2,1,1}},
    {0xeb67,0xeb69,{1,1,1,1,2,2,1}},
    {0xeb6a,0xeb6e,{1,1,1,1,2,1,1}},
    {0xeb6f,0xeb71,{1,1,1,1,2,2,1}},
    {0xeb72,0xeb72,{1,1,1,1,2,1,1}},
    {0xeb73,0xeb73,{1,1,1,1,2,2,1}},
    {0xeb74,0xeb75,{1,1,1,1,2,1,1}},
    {0xeb76,0xeb77,{1,1,1,1,2,2,1}},
    {0xeb78,0xeb79,{1,1,1,1,2,1,1}},
    {0xeb7a,0xeb7a,{1,1,1,1,2,2,1}},
    {0xeb7b,0xeb7d,{1,1,1,1,2,1,1}},
    {0xeb7e,0xeb80,{1,1,1,1,2,2,1}},
    {0xeb81,0xeb82,{1,1,1,1,2,1,1}},
    {0xeb83,0xeb83,{1,1,1,1,2,2,1}},
    {0xeb84,0xeb87,{1,1,1,1,2,1,1}},
    {0xeb88,0xeb88,{1,1,1,1,2,2,1}},
    {0xeb89,0xeb89,{1,1,1,1,2,1,1}},
    {0xeb8a,0xeb8a,{1,1,1,1,2,2,1}},
    {0xeb8b,0xeb8b,{1,1,1,1,2,1,1}},
    {0xeb8c,0xeb8c,{1,1,1,1,2,2,1}},
    {0xeb8d,0xeb8f,{1,1,1,1,2,1,1}},
    {0xeb90,0xeb90,{1,1,1,1,2,2,1}},
    {0xeb91,0xeb92,{1,1,1,1,2,1,1}},
    {0xeb93,0xeb93,{1,1,1,1,2,2,1}},
    {0xeb94,0xeb94,{1,1,1,1,2,1,1}},
    {0xeb95,0xeb95,{1,1,1,1,2,2,1}},
    {0xeb96,0xeb98,{1,1,1,1,2,1,1}},
    {0xeb99,0xeb99,{1,1,1,1,2,2,1}},
    {0xeb9a,0xeb9a,{1,1,1,1,2,1,1}},
    {0xeb9b,0xeb9b,{1,1,1,1,2,2,1}},
    {0xeb9c,0xeb9e,{1,1,1,1,2,1,1}},
    {0xeb9f,0xeb9f,{1,1,1,1,2,2,1}},
    {0xeba0,0xeba1,{1,1,1,1,2,1,1}},
    {0xeba2,0xeba2,{1,1,1,1,2,2,1}},
    {0xeba3,0xeba4,{1,1,1,1,2,1,1}},
    {0xeba5,0xeba6,{1,1,1,1,2,2,1}},
    {0xeba7,0xeba9,{1,1,1,1,2,1,1}},
    {0xebaa,0xebab,{1,1,1,1,2,2,1}},
    {0xebac,0xebad,{1,1,1,1,2,1,1}},
    {0xebae,0xebae,{1,1,1,1,2,2,1}},
    {0xebaf,0xebb1,{1,1,1,1,2,1,1}},
    {0xebb2,0xebb3,{1,1,1,1,2,2,1}},
    {0xebb4,0xebbc,{1,1,1,1,2,1,1}},
    {0xebbd,0xebbd,{1,1,1,1,2,2,1}},
    {0xebbe,0xebbe,{1,1,1,1,2,1,1}},
    {0xebbf,0xebbf,{1,1,1,1,2,2,1}},
    {0xebc0,0xebc4,{1,1,1,1,2,1,1}},
    {0xebc5,0xebc5,{1,1,1,1,2,2,1}},
    {0xebc6,0xebc9,{1,1,1,1,2,1,1}},
    {0xebca,0xebca,{1,1,1,1,2,2,1}},
    {0xebcb,0xebcb,{1,1,1,1,2,1,1}},
    {0xebcc,0xebcc,{1,1,1,1,2,2,1}},
    {0xebcd,0xebcd,{1,1,1,1,2,1,1}},
    {0xebce,0xebce,{1,1,1,1,2,2,1}},
    {0xebcf,0xebcf,{1,1,1,1,2,1,1}},
    {0xebd0,0xebd0,{1,1,1,1,2,2,1}},
    {0xebd1,0xebd4,{1,1,1,1,2,1,1}},
    {0xebd5,0xebd5,{1,1,1,1,2,2,1}},
    {0xebd6,0xebd6,{1,1,1,1,2,1,1}},
    {0xebd7,0xebd8,{1,1,1,1,2,2,1}},
    {0xebd9,0xebd9,{1,1,1,1,2,1,1}},
    {0xebda,0xebdb,{1,1,1,1,2,2,1}},
    {0xebdc,0xebdc,{1,1,1,1,2,1,1}},
    {0xebdd,0xebdd,{1,1,1,1,2,2,1}},
    {0xebde,0xebde,{1,1,1,1,2,1,1}},
    {0xebdf,0xebdf,{1,1,1,1,2,2,1}},
    {0xebe0,0xebe0,{1,1,1,1,2,1,1}},
    {0xebe1,0xebe1,{1,1,1,1,2,2,1}},
    {0xebe2,0xebe3,{1,1,1,1,2,1,1}},
    {0xebe4,0xebe4,{1,1,1,1,2,2,1}},
    {0xebe5,0xebe7,{1,1,1,1,2,1,1}},
    {0xebe8,0xebe9,{1,1,1,1,2,2,1}},
    {0xebea,0xebea,{1,1,1,1,2,1,1}},
    {0xebeb,0xebec,{1,1,1,1,2,2,1}},
    {0xebed,0xebee,{1,1,1,1,2,1,1}},
    {0xebef,0xebef,{1,1,1,1,2,2,1}},
    {0xebf0,0xebf1,{1,1,1,1,2,1,1}},
    {0xebf2,0xebf2,{1,1,1,1,2,2,1}},
    {0xebf3,0xebf4,{1,1,1,1,2,1,1}},
    {0xebf5,0xebf5,{1,1,1,1,2,2,1}},
    {0xebf6,0xebf8,{1,1,1,1,2,1,1}},
    {0xebf9,0xebfa,{1,1,1,1,2,2,1}},
    {0xebfb,0xebfd,{1,1,1,1,2,1,1}},
    {0xebfe,0xebfe,{1,1,1,1,2,2,1}},
    {0xebff,0xec07,{1,1,1,1,2,1,1}},
    {0xec08,0xec09,{1,1,1,1,2,2,1}},
    {0xec0a,0xec0e,{1,1,1,1,2,1,1}},
    {0xec0f,0xec0f,{1,1,1,1,2,2,1}},
    {0xec10,0xec12,{1,1,1,1,2,1,1}},
    {0xec13,0xec13,{1,1,1,1,2,2,1}},
    {0xec14,0xec18,{1,1,1,1,2,1,1}},
    {0xec19,0xec1c,{1,1,1,1,2,2,1}},
    {0xec1d,0xec23,{1,1,1,1,2,1,1}},
    {0xec24,0xec24,{1,1,1,1,2,2,1}},
    {0xec25,0xec25,{1,1,1,1,2,1,1}},
    {0xec26,0xec26,{1,1,1,1,2,2,1}},
    {0xec27,0xec2a,{1,1,1,1,2,1,1}},
    {0xec2b,0xec2b,{1,1,1,1,2,2,1}},
    {0xec2c,0xec2e,{1,1,1,1,2,1,1}},
    {0xec2f,0xec2f,{1,1,1,1,2,2,1}},
    {0xec30,0xec31,{1,1,1,1,2,1,1}},
    {0xec32,0xec32,{1,1,1,1,2,2,1}},
    {0xec33,0xec3e,{1,1,1,1,2,1,1}},
    {0xec3f,0xec3f,{1,1,1,1,2,2,1}},
    {0xec40,0xec42,{1,1,1,1,2,1,1}},
    {0xec43,0xec49,{1,1,1,1,2,2,1}},
    {0xec4a,0xec4a,{1,1,1,1,2,1,1}},
    {0xec4b,0xec4b,{1,1,1,1,2,2,1}},
    {0xec4c,0xec4e,{1,1,1,1,2,1,1}},
    {0xec4f,0xec4f,{1,1,1,1,2,2,1}},
    {0xec50,0xec51,{1,1,1,1,2,1,1}},
    {0xec52,0xec52,{1,1,1,1,2,2,1}},
    {0xec53,0xec53,{1,1,1,1,2,1,1}},
    {0xec54,0xec54,{1,1,1,1,2,2,1}},
    {0xec55,0xec55,{1,1,1,1,2,1,1}},
    {0xec56,0xec58,{1,1,1,1,2,2,1}},
    {0xec59,0xec59,{1,1,1,1,2,1,1}},
    {0xec5a,0xec5a,{1,1,1,1,2,2,1}},
    {0xec5b,0xec5b,{1,1,1,1,2,1,1}},
    {0xec5c,0xec5d,{1,1,1,1,2,2,1}},
    {0xec5e,0xec5f,{1,1,1,1,2,1,1}},
    {0xec60,0xec60,{1,1,1,1,2,2,1}},
    {0xec61,0xec61,{1,1,1,1,2,1,1}},
    {0xec62,0xec63,{1,1,1,1,2,2,1}},
    {0xec64,0xec66,{1,1,1,1,2,1,1}},
    {0xec67,0xec67,{1,1,1,1,2,2,1}},
    {0xec68,0xec68,{1,1,1,1,2,1,1}},
    {0xec69,0xec6d,{1,1,1,1,2,2,1}},
    {0xec6e,0xec6e,{1,1,1,1,2,1,1}},
    {0xec6f,0xec70,{1,1,1,1,2,2,1}},
    {0xec71,0xec71,{1,1,1,1,2,1,1}},
    {0xec72,0xec73,{1,1,1,1,2,2,1}},
    {0xec74,0xec74,{1,1,1,1,2,1,1}},
    {0xec75,0xec75,{1,1,1,1,2,2,1}},
    {0xec76,0xec78,{1,1,1,1,2,1,1}},
    {0xec79,0xec79,{1,1,1,1,2,2,1}},
    {0xec7a,0xec7a,{1,1,1,1,2,1,1}},
    {0xec7b,0xec7b,{1,1,1,1,2,2,1}},
    {0xec7c,0xec7d,{1,1,1,1,2,1,1}},
    {0xec7e,0xec7e,{1,1,1,1,2,2,1}},
    {0xec7f,0xec83,{1,1,1,1,2,1,1}},
    {0xec84,0xec84,{1,1,1,1,2,2,1}},
    {0xec85,0xec8d,{1,1,1,1,2,1,1}},
    {0xec8e,0xec8e,{1,1,1,1,2,2,1}},
    {0xec8f,0xec91,{1,1,1,1,2,1,1}},
    {0xec92,0xec92,{1,1,1,1,2,2,1}},
    {0xec93,0xec94,{1,1,1,1,2,1,1}},
    {0xec95,0xec95,{1,1,1,1,2,2,1}},
    {0xec96,0xecae,{1,1,1,1,2,1,1}},
    {0xecaf,0xecaf,{1,1,1,1,2,2,1}},
    {0xecb0,0xecb5,{1,1,1,1,2,1,1}},
    {0xecb6,0xecb6,{1,1,1,1,2,2,1}},
    {0xecb7,0xecce,{1,1,1,1,2,1,1}},
    {0xeccf,0xeccf,{1,1,1,1,2,2,1}},
    {0xecd0,0xecdb,{1,1,1,1,2,1,1}},
    {0xecdc,0xecdc,{1,1,1,1,2,2,1}},
    {0xecdd,0xecdf,{1,1,1,1,2,1,1}},
    {0xece0,0xece1,{1,1,1,1,2,2,1}},
    {0xece2,0xecef,{1,1,1,1,2,1,1}},
    {0xecf0,0xecf0,{1,1,1,1,2,2,1}},
    {0xecf1,0xecf1,{1,1,1,1,2,1,1}},
    {0xecf2,0xecf2,{1,1,1,1,2,2,1}},
    {0xecf3,0xecfb,{1,1,1,1,2,1,1}},
    {0xecfc,0xecfc,{1,1,1,1,2,2,1}},
    {0xecfd,0xed10,{1,1,1,1,2,1,1}},
    {0xed11,0xed11,{1,1,1,1,2,2,1}},
    {0xed12,0xed1c,{1,1,1,1,2,1,1}},
    {0xed1d,0xed1d,{1,1,1,1,2,2,1}},
    {0xed1e,0xed23,{1,1,1,1,2,1,1}},
    {0xed24,0xed24,{1,1,1,1,2,2,1}},
    {0xed25,0xed28,{1,1,1,1,2,1,1}},
    {0xed29,0xed29,{1,1,1,1,2,2,1}},
    {0xed2a,0xed2b,{1,1,1,1,2,1,1}},
    {0xed2c,0xed2c,{1,1,1,1,2,2,1}},
    {0xed2d,0xed3a,{1,1,1,1,2,1,1}},
    {0xed3b,0xed3b,{1,1,1,1,2,2,1}},
    {0xed3c,0xed40,{1,1,1,1,2,1,1}},
    {0xed41,0xed41,{1,1,1,1,2,2,1}},
    {0xed42,0xed4e,{1,1,1,1,2,1,1}},
    {0xed4f,0xed4f,{1,1,1,1,2,2,1}},
    {0xed50,0xed51,{1,1,1,1,2,1,1}},
    {0xed52,0xed52,{1,1,1,1,2,2,1}},
    {0xed53,0xed5e,{1,1,1,1,2,1,1}},
    {0xed5f,0xed5f,{1,1,1,1,2,2,1}},
    {0xed60,0xed6a,{1,1,1,1,2,1,1}},
    {0xed6b,0xed6b,{1,1,1,1,2,2,1}},
    {0xed6c,0xed77,{1,1,1,1,2,1,1}},
    {0xed78,0xed78,{1,1,1,1,2,2,1}},
    {0xed79,0xed83,{1,1,1,1,2,1,1}},
    {0xed84,0xed85,{1,1,1,1,2,2,1}},
    {0xed86,0xed87,{1,1,1,1,2,1,1}},
    {0xed88,0xed88,{1,1,1,1,2,2,1}},
    {0xed89,0xed8a,{1,1,1,1,2,1,1}},
    {0xed8b,0xed8b,{1,1,1,1,2,2,1}},
    {0xed8c,0xed8d,{1,1,1,1,2,1,1}},
    {0xed8e,0xed8e,{1,1,1,1,2,2,1}},
    {0xed8f,0xedce,{1,1,1,1,2,1,1}},
    {0xedcf,0xedcf,{1,1,1,1,2,2,1}},
    {0xedd0,0xedd9,{1,1,1,1,2,1,1}},
    {0xedda,0xedda,{1,1,1,1,2,2,1}},
    {0xeddb,0xedec,{1,1,1,1,2,1,1}},
    {0xeded,0xeded,{1,1,1,1,2,2,1}},
    {0xedee,0xee07,{1,1,1,1,2,1,1}},
    {0xee08,0xee08,{1,1,1,1,2,2,1}},
    {0xee09,0xee5c,{1,1,1,1,2,1,1}},
    {0xee5d,0xee5d,{1,1,1,1,2,2,1}},
    {0xee5e,0xee69,{1,1,1,1,2,1,1}},
    {0xee6a,0xee6a,{1,1,1,1,2,2,1}},
    {0xee6b,0xeea4,{1,1,1,1,2,1,1}},
    {0xeea5,0xeea5,{1,1,1,1,2,2,1}},
    {0xeea6,0xeeaf,{1,1,1,1,2,1,1}},
    {0xeeb0,0xeeb0,{1,1,1,1,2,2,1}},
    {0xeeb1,0xeeb3,{1,1,1,1,2,1,1}},
    {0xeeb4,0xeeb4,{1,1,1,1,2,2,1}},
    {0xeeb5,0xeeb6,{1,1,1,1,2,1,1}},
    {0xeeb7,0xeeb7,{1,1,1,1,2,2,1}},
    {0xeeb8,0xf302,{1,1,1,1,2,1,1}},
    {0xf303,0xf318,{1,1,1,1,2,2,1}},
    {0xf319,0xf324,{1,1,1,1,2,1,1}},
    {0xf325,0xf325,{1,1,1,1,2,2,1}},
    {0xf326,0xf326,{1,1,1,1,2,1,1}},
    {0xf327,0xf327,{1,1,1,1,2,2,1}},
    {0xf328,0xf343,{1,1,1,1,2,1,1}},
    {0xf344,0xf344,{1,1,1,1,2,2,1}},
    {0xf345,0xf345,{1,1,1,1,2,1,1}},
    {0xf346,0xf346,{1,1,1,1,2,2,1}},
    {0xf347,0xf349,{1,1,1,1,2,1,1}},
    {0xf34a,0xf34b,{1,1,1,1,2,2,1}},
    {0xf34c,0xf3a0,{1,1,1,1,2,1,1}},
    {0xf3a1,0xf3a1,{1,1,1,1,2,2,1}},
    {0xf3a2,0xf3ef,{1,1,1,1,2,1,1}},
    {0xf3f0,0xf3f0,{1,1,1,1,2,2,1}},
    {0xf3f1,0xf3f8,{1,1,1,1,2,1,1}},
    {0xf3f9,0xf3f9,{1,1,1,1,2,2,1}},
    {0xf3fa,0xf3fb,{1,1,1,1,2,1,1}},
    {0xf3fc,0xf3fc,{1,1,1,1,2,2,1}},
    {0xf3fd,0xf407,{1,1,1,1,2,1,1}},
    {0xf408,0xf408,{1,1,1,1,2,2,1}},
    {0xf409,0xf40a,{1,1,1,1,2,1,1}},
    {0xf40b,0xf40b,{1,1,1,1,2,2,1}},
    {0xf40c,0xf43c,{1,1,1,1,2,1,1}},
    {0xf43d,0xf43d,{1,1,1,1,2,2,1}},
    {0xf43e,0xf440,{1,1,1,1,2,1,1}},
    {0xf441,0xf442,{1,1,1,1,2,2,1}},
    {0xf443,0xf448,{1,1,1,1,2,1,1}},
    {0xf449,0xf449,{1,1,1,1,2,2,1}},
    {0xf44a,0xf44c,{1,1,1,1,2,1,1}},
    {0xf44d,0xf44d,{1,1,1,1,2,2,1}},
    {0xf44e,0xf44f,{1,1,1,1,2,1,1}},
    {0xf450,0xf450,{1,1,1,1,2,2,1}},
    {0xf451,0xf453,{1,1,1,1,2,1,1}},
    {0xf454,0xf454,{1,1,1,1,2,2,1}},
    {0xf455,0xf459,{1,1,1,1,2,1,1}},
    {0xf45a,0xf45d,{1,1,1,1,2,2,1}},
    {0xf45e,0xf461,{1,1,1,1,2,1,1}},
    {0xf462,0xf462,{1,1,1,1,2,2,1}},
    {0xf463,0xf464,{1,1,1,1,2,1,1}},
    {0xf465,0xf466,{1,1,1,1,2,2,1}},
    {0xf467,0xf468,{1,1,1,1,2,1,1}},
    {0xf469,0xf469,{1,1,1,1,2,2,1}},
    {0xf46a,0xf46c,{1,1,1,1,2,1,1}},
    {0xf46d,0xf46f,{1,1,1,1,2,2,1}},
    {0xf470,0xf476,{1,1,1,1,2,1,1}},
    {0xf477,0xf478,{1,1,1,1,2,2,1}},
    {0xf479,0xf47b,{1,1,1,1,2,1,1}},
    {0xf47c,0xf47e,{1,1,1,1,2,2,1}},
    {0xf47f,0xf47f,{1,1,1,1,2,1,1}},
    {0xf480,0xf480,{1,1,1,1,2,2,1}},
    {0xf481,0xf484,{1,1,1,1,2,1,1}},
    {0xf485,0xf485,{1,1,1,1,2,2,1}},
    {0xf486,0xf487,{1,1,1,1,2,1,1}},
    {0xf488,0xf488,{1,1,1,1,2,2,1}},
    {0xf489,0xf48a,{1,1,1,1,2,1,1}},
    {0xf48b,0xf48b,{1,1,1,1,2,2,1}},
    {0xf48c,0xf48c,{1,1,1,1,2,1,1}},
    {0xf48d,0xf48d,{1,1,1,1,2,2,1}},
    {0xf48e,0xf48e,{1,1,1,1,2,1,1}},
    {0xf48f,0xf491,{1,1,1,1,2,2,1}},
    {0xf492,0xf496,{1,1,1,1,2,1,1}},
    {0xf497,0xf498,{1,1,1,1,2,2,1}},
    {0xf499,0xf49a,{1,1,1,1,2,1,1}},
    {0xf49b,0xf49b,{1,1,1,1,2,2,1}},
    {0xf49c,0xf4a3,{1,1,1,1,2,1,1}},
    {0xf4a4,0xf4a4,{1,1,1,1,2,2,1}},
    {0xf4a5,0xf4a6,{1,1,1,1,2,1,1}},
    {0xf4a7,0xf4a7,{1,1,1,1,2,2,1}},
    {0xf4a8,0xf4a9,{1,1,1,1,2,1,1}},
    {0xf4aa,0xf4aa,{1,1,1,1,2,2,1}},
    {0xf4ab,0xf4ac,{1,1,1,1,2,1,1}},
    {0xf4ad,0xf4ad,{1,1,1,1,2,2,1}},
    {0xf4ae,0xf4ae,{1,1,1,1,2,1,1}},
    {0xf4af,0xf4af,{1,1,1,1,2,2,1}},
    {0xf4b0,0xf4b2,{1,1,1,1,2,1,1}},
    {0xf4b3,0xf4b4,{1,1,1,1,2,2,1}},
    {0xf4b5,0xf4b6,{1,1,1,1,2,1,1}},
    {0xf4b7,0xf4b7,{1,1,1,1,2,2,1}},
    {0xf4b8,0xf4b9,{1,1,1,1,2,1,1}},
    {0xf4ba,0xf4bd,{1,1,1,1,2,2,1}},
    {0xf4be,0xf4bf,{1,1,1,1,2,1,1}},
    {0xf4c0,0xf4c1,{1,1,1,1,2,2,1}},
    {0xf4c2,0xf4c2,{1,1,1,1,2,1,1}},
    {0xf4c3,0xf4c3,{1,1,1,1,2,2,1}},
    {0xf4c4,0xf4c7,{1,1,1,1,2,1,1}},
    {0xf4c8,0xf4c8,{1,1,1,1,2,2,1}},
    {0xf4c9,0xf4cd,{1,1,1,1,2,1,1}},
    {0xf4ce,0xf4cf,{1,1,1,1,2,2,1}},
    {0xf4d0,0xf4d1,{1,1,1,1,2,1,1}},
    {0xf4d2,0xf4d2,{1,1,1,1,2,2,1}},
    {0xf4d3,0xf4e1,{1,1,1,1,2,1,1}},
    {0xf4e2,0xf4e2,{1,1,1,1,2,2,1}},
    {0xf4e3,0xf4e4,{1,1,1,1,2,1,1}},
    {0xf4e5,0xf4e5,{1,1,1,1,2,2,1}},
    {0xf4e6,0xf4e6,{1,1,1,1,2,1,1}},
    {0xf4e7,0xf4e9,{1,1,1,1,2,2,1}},
    {0xf4ea,0xf4f2,{1,1,1,1,2,1,1}},
    {0xf4f3,0xf4f3,{1,1,1,1,2,2,1}},
    {0xf4f4,0xf521,{1,1,1,1,2,1,1}},
    {0xf522,0xf523,{1,1,1,1,2,2,1}},
    {0xf524,0xf524,{1,1,1,1,2,1,1}},
    {0xf525,0xf526,{1,1,1,1,2,2,1}},
    {0xf527,0xf529,{1,1,1,1,2,1,1}},
    {0xf52a,0xf52a,{1,1,1,1,2,2,1}},
    {0xf52b,0xf539,{1,1,1,1,2,1,1}},
    {0xf53a,0xf53b,{1,1,1,1,2,2,1}},
    {0xf53c,0xf53c,{1,1,1,1,2,1,1}},
    {0xf53d,0xf53d,{1,1,1,1,2,2,1}},
    {0xf53e,0xf54d,{1,1,1,1,2,1,1}},
    {0xf54e,0xf54e,{1,1,1,1,2,2,1}},
    {0xf54f,0xf552,{1,1,1,1,2,1,1}},
    {0xf553,0xf553,{1,1,1,1,2,2,1}},
    {0xf554,0xf555,{1,1,1,1,2,1,1}},
    {0xf556,0xf556,{1,1,1,1,2,2,1}},
    {0xf557,0xf558,{1,1,1,1,2,1,1}},
    {0xf559,0xf55a,{1,1,1,1,2,2,1}},
    {0xf55b,0xf55d,{1,1,1,1,2,1,1}},
    {0xf55e,0xf55e,{1,1,1,1,2,2,1}},
    {0xf55f,0xf564,{1,1,1,1,2,1,1}},
    {0xf565,0xf565,{1,1,1,1,2,2,1}},
    {0xf566,0xf56a,{1,1,1,1,2,1,1}},
    {0xf56b,0xf56b,{1,1,1,1,2,2,1}},
    {0xf56c,0xf571,{1,1,1,1,2,1,1}},
    {0xf572,0xf572,{1,1,1,1,2,2,1}},
    {0xf573,0xf6af,{1,1,1,1,2,1,1}},
    {0xf6b0,0xf6b0,{1,1,1,1,2,2,1}},
    {0xf6b1,0xf7e4,{1,1,1,1,2,1,2}},
    {0xf7e5,0xf7e6,{1,1,1,1,2,2,2}},
    {0xf7e7,0xf7e7,{1,1,1,1,2,1,2}},
    {0xf7e8,0xf7e8,{1,1,1,1,2,2,2}},
    {0xf7e9,0xf7e9,{1,1,1,1,2,1,2}},
    {0xf7ea,0xf7eb,{1,1,1,1,2,2,2}},
    {0xf7ec,0xf7ec,{1,1,1,1,2,1,2}},
    {0xf7ed,0xf7ee,{1,1,1,1,2,2,2}},
    {0xf7ef,0xf848,{1,1,1,1,2,1,2}},  // EA ambiguous to f8ff
    {0xf849,0xf8ef,{1,1,1,1,2,1,1}},
    {0xf900,0xfaff,{2,2,2,2,2,2,2}}, // CJK COMPATIBILITY IDEOGRAPH
    {0xfe10,0xfe19,{2,2,2,2,2,2,2}}, // Presentation forms, small signs
    {0xfe30,0xfe52,{2,2,2,2,2,2,2}},
    {0xfe54,0xfe66,{2,2,2,2,2,2,2}},
    {0xfe68,0xfe6b,{2,2,2,2,2,2,2}}, // end of small signs
    {0xff01,0xff60,{2,2,2,2,2,2,2}}, // fullwidth
    {0xff64,0xff64,{1,1,1,1,1,2,1}},
    {0xffe0,0xffe6,{2,2,2,2,2,2,2}}, // fullwidth
    {0xfffd,0xfffd,{1,1,1,1,2,1,1}}, // EA ambiguous
    {0x16fe0,0x16fe3,{2,2,2,2,2,2,2}},
    {0x16ff0,0x16ff1,{2,2,2,2,2,2,2}},
    {0x17000,0x187f7,{2,2,2,2,2,2,2}},
    {0x18800,0x18cd5,{2,2,2,2,2,2,2}},
    {0x18d00,0x18d08,{2,2,2,2,2,2,2}},
    {0x1b000,0x1b11e,{2,2,2,2,2,2,2}},
    {0x1b150,0x1b152,{2,2,2,2,2,2,2}},
    {0x1b164,0x1b167,{2,2,2,2,2,2,2}},
    {0x1b170,0x1b2fb,{2,2,2,2,2,2,2}},
    {0x1f004,0x1f004,{2,2,2,2,2,2,2}},
    {0x1f0cf,0x1f0cf,{2,2,2,2,2,2,2}},
    {0x1f18e,0x1f18e,{2,2,2,2,2,2,2}},
    {0x1f191,0x1f19a,{2,2,2,2,2,2,2}},
    {0x1f200,0x1f202,{2,2,2,2,2,2,2}},
    {0x1f210,0x1f23b,{2,2,2,2,2,2,2}},
    {0x1f240,0x1f248,{2,2,2,2,2,2,2}},
    {0x1f250,0x1f251,{2,2,2,2,2,2,2}},
    {0x1f260,0x1f265,{2,2,2,2,2,2,2}},
    {0x1f300,0x1f320,{2,2,2,2,2,2,2}},
    {0x1f32d,0x1f335,{2,2,2,2,2,2,2}},
    {0x1f337,0x1f37c,{2,2,2,2,2,2,2}},
    {0x1f37e,0x1f393,{2,2,2,2,2,2,2}},
    {0x1f3a0,0x1f3ca,{2,2,2,2,2,2,2}},
    {0x1f3cf,0x1f3d3,{2,2,2,2,2,2,2}},
    {0x1f3e0,0x1f3f0,{2,2,2,2,2,2,2}},
    {0x1f3f4,0x1f3f4,{2,2,2,2,2,2,2}},
    {0x1f3f8,0x1f43e,{2,2,2,2,2,2,2}},
    {0x1f440,0x1f440,{2,2,2,2,2,2,2}},
    {0x1f442,0x1f4fc,{2,2,2,2,2,2,2}},
    {0x1f4ff,0x1f53d,{2,2,2,2,2,2,2}},
    {0x1f54b,0x1f54e,{2,2,2,2,2,2,2}},
    {0x1f550,0x1f567,{2,2,2,2,2,2,2}},
    {0x1f57a,0x1f57a,{2,2,2,2,2,2,2}},
    {0x1f595,0x1f596,{2,2,2,2,2,2,2}},
    {0x1f5a4,0x1f5a4,{2,2,2,2,2,2,2}},
    {0x1f5fb,0x1f64f,{2,2,2,2,2,2,2}},
    {0x1f680,0x1f6c5,{2,2,2,2,2,2,2}},
    {0x1f6cc,0x1f6cc,{2,2,2,2,2,2,2}},
    {0x1f6d0,0x1f6d2,{2,2,2,2,2,2,2}},
    {0x1f6d5,0x1f6d7,{2,2,2,2,2,2,2}},
    {0x1f6eb,0x1f6ec,{2,2,2,2,2,2,2}},
    {0x1f6f4,0x1f6fc,{2,2,2,2,2,2,2}},
    {0x1f7e0,0x1f7eb,{2,2,2,2,2,2,2}},
    {0x1f90c,0x1f93a,{2,2,2,2,2,2,2}},
    {0x1f93c,0x1f945,{2,2,2,2,2,2,2}},
    {0x1f947,0x1f978,{2,2,2,2,2,2,2}},
    {0x1f97a,0x1f9cb,{2,2,2,2,2,2,2}},
    {0x1f9cd,0x1f9ff,{2,2,2,2,2,2,2}},
    {0x1fa70,0x1fa74,{2,2,2,2,2,2,2}},
    {0x1fa78,0x1fa7a,{2,2,2,2,2,2,2}},
    {0x1fa80,0x1fa86,{2,2,2,2,2,2,2}},
    {0x1fa90,0x1faa8,{2,2,2,2,2,2,2}},
    {0x1fab0,0x1fab6,{2,2,2,2,2,2,2}},
    {0x1fac0,0x1fac2,{2,2,2,2,2,2,2}},
    {0x1fad0,0x1fad6,{2,2,2,2,2,2,2}},
    // 13.0.0 has all in planes 2 and 3 as wide chars, assigned or not
    {0x20000,0x3ffff,{2,2,2,2,2,2,2}}
};

/* From http://www.cl.cam.ac.uk/~mgk25/ucs/wcwidth.c
 *    <quote>
 *    - The null character (U+0000) has a column width of 0.
 *
 *    - Other C0/C1 control characters and DEL will lead to a return
 *      value of -1.
 *
 *    - Non-spacing and enclosing combining characters (general
 *      category code Mn or Me in the Unicode database) have a
 *      column width of 0.
 *
 *    - SOFT HYPHEN (U+00AD) has a column width of 1.
 *
 *    - Prepended_Concatenation_Mark have column width of 1.
 *
 *    - Other format characters (general category code Cf in the Unicode
 *      database) and ZERO WIDTH SPACE (U+200B) have a column width of 0.
 *
 *    - Hangul Jamo medial vowels and final consonants (U+1160-U+11FF)
 *      have a column width of 0.
 *    </quote>
 *
 *     But this version has
 *    - C0 and C1 control characters have a column width of 0.
 *
 *    Unicode has U+00AD as a Cf character. which would imply width 0.
 *    U+1160-11FF are Lo ('letter, other'): glibc >= 2.27 agrees
 *
 *    The original wcwidth.c ays
 *    generated by "uniset +cat=Me +cat=Mn +cat=Cf -00AD +1160-11FF +200B c"
 *
 *    200B ('zero width space') is in Cf, so does not need to be an exception.
 *
 *    2020-05: updated based on the Unicode 12.1.0 tables at
 *    https://www.unicode.org/Public/12.1.0/ucd/UnicodeData.txt
 *    https://www.unicode.org/Public/12.1.0/ucd/EastAsianWidth.txt
 *
 *    2020-12: uniset re-run on current version (13.0.0)
 *    https://www.unicode.org/Public/UCD/latest/ucd/UnicodeData.txt
 *    still adding the first two ranges.
 *
 */

static const struct interval zero_width[] = {
    { 0x0000, 0x001F }, { 0x007F, 0x009F },
    { 0x0300, 0x036F }, { 0x0483, 0x0489 }, { 0x0591, 0x05BD },
    { 0x05BF, 0x05BF }, { 0x05C1, 0x05C2 }, { 0x05C4, 0x05C5 },
    { 0x05C7, 0x05C7 }, { 0x0600, 0x0605 }, { 0x0610, 0x061A },
    { 0x061C, 0x061C }, { 0x064B, 0x065F }, { 0x0670, 0x0670 },
    { 0x06D6, 0x06DD }, { 0x06DF, 0x06E4 }, { 0x06E7, 0x06E8 },
    { 0x06EA, 0x06ED }, { 0x070F, 0x070F }, { 0x0711, 0x0711 },
    { 0x0730, 0x074A }, { 0x07A6, 0x07B0 }, { 0x07EB, 0x07F3 },
    { 0x07FD, 0x07FD }, { 0x0816, 0x0819 }, { 0x081B, 0x0823 },
    { 0x0825, 0x0827 }, { 0x0829, 0x082D }, { 0x0859, 0x085B },
    { 0x08D3, 0x0902 }, { 0x093A, 0x093A }, { 0x093C, 0x093C },
    { 0x0941, 0x0948 }, { 0x094D, 0x094D }, { 0x0951, 0x0957 },
    { 0x0962, 0x0963 }, { 0x0981, 0x0981 }, { 0x09BC, 0x09BC },
    { 0x09C1, 0x09C4 }, { 0x09CD, 0x09CD }, { 0x09E2, 0x09E3 },
    { 0x09FE, 0x09FE }, { 0x0A01, 0x0A02 }, { 0x0A3C, 0x0A3C },
    { 0x0A41, 0x0A42 }, { 0x0A47, 0x0A48 }, { 0x0A4B, 0x0A4D },
    { 0x0A51, 0x0A51 }, { 0x0A70, 0x0A71 }, { 0x0A75, 0x0A75 },
    { 0x0A81, 0x0A82 }, { 0x0ABC, 0x0ABC }, { 0x0AC1, 0x0AC5 },
    { 0x0AC7, 0x0AC8 }, { 0x0ACD, 0x0ACD }, { 0x0AE2, 0x0AE3 },
    { 0x0AFA, 0x0AFF }, { 0x0B01, 0x0B01 }, { 0x0B3C, 0x0B3C },
    { 0x0B3F, 0x0B3F }, { 0x0B41, 0x0B44 }, { 0x0B4D, 0x0B4D },
    { 0x0B55, 0x0B56 }, { 0x0B62, 0x0B63 }, { 0x0B82, 0x0B82 },
    { 0x0BC0, 0x0BC0 }, { 0x0BCD, 0x0BCD }, { 0x0C00, 0x0C00 },
    { 0x0C04, 0x0C04 }, { 0x0C3E, 0x0C40 }, { 0x0C46, 0x0C48 },
    { 0x0C4A, 0x0C4D }, { 0x0C55, 0x0C56 }, { 0x0C62, 0x0C63 },
    { 0x0C81, 0x0C81 }, { 0x0CBC, 0x0CBC }, { 0x0CBF, 0x0CBF },
    { 0x0CC6, 0x0CC6 }, { 0x0CCC, 0x0CCD }, { 0x0CE2, 0x0CE3 },
    { 0x0D00, 0x0D01 }, { 0x0D3B, 0x0D3C }, { 0x0D41, 0x0D44 },
    { 0x0D4D, 0x0D4D }, { 0x0D62, 0x0D63 }, { 0x0D81, 0x0D81 },
    { 0x0DCA, 0x0DCA }, { 0x0DD2, 0x0DD4 }, { 0x0DD6, 0x0DD6 },
    { 0x0E31, 0x0E31 }, { 0x0E34, 0x0E3A }, { 0x0E47, 0x0E4E },
    { 0x0EB1, 0x0EB1 }, { 0x0EB4, 0x0EBC }, { 0x0EC8, 0x0ECD },
    { 0x0F18, 0x0F19 }, { 0x0F35, 0x0F35 }, { 0x0F37, 0x0F37 },
    { 0x0F39, 0x0F39 }, { 0x0F71, 0x0F7E }, { 0x0F80, 0x0F84 },
    { 0x0F86, 0x0F87 }, { 0x0F8D, 0x0F97 }, { 0x0F99, 0x0FBC },
    { 0x0FC6, 0x0FC6 }, { 0x102D, 0x1030 }, { 0x1032, 0x1037 },
    { 0x1039, 0x103A }, { 0x103D, 0x103E }, { 0x1058, 0x1059 },
    { 0x105E, 0x1060 }, { 0x1071, 0x1074 }, { 0x1082, 0x1082 },
    { 0x1085, 0x1086 }, { 0x108D, 0x108D }, { 0x109D, 0x109D },
    { 0x1160, 0x11FF }, { 0x135D, 0x135F }, { 0x1712, 0x1714 },
    { 0x1732, 0x1734 }, { 0x1752, 0x1753 }, { 0x1772, 0x1773 },
    { 0x17B4, 0x17B5 }, { 0x17B7, 0x17BD }, { 0x17C6, 0x17C6 },
    { 0x17C9, 0x17D3 }, { 0x17DD, 0x17DD }, { 0x180B, 0x180E },
    { 0x1885, 0x1886 }, { 0x18A9, 0x18A9 }, { 0x1920, 0x1922 },
    { 0x1927, 0x1928 }, { 0x1932, 0x1932 }, { 0x1939, 0x193B },
    { 0x1A17, 0x1A18 }, { 0x1A1B, 0x1A1B }, { 0x1A56, 0x1A56 },
    { 0x1A58, 0x1A5E }, { 0x1A60, 0x1A60 }, { 0x1A62, 0x1A62 },
    { 0x1A65, 0x1A6C }, { 0x1A73, 0x1A7C }, { 0x1A7F, 0x1A7F },
    { 0x1AB0, 0x1AC0 }, { 0x1B00, 0x1B03 }, { 0x1B34, 0x1B34 },
    { 0x1B36, 0x1B3A }, { 0x1B3C, 0x1B3C }, { 0x1B42, 0x1B42 },
    { 0x1B6B, 0x1B73 }, { 0x1B80, 0x1B81 }, { 0x1BA2, 0x1BA5 },
    { 0x1BA8, 0x1BA9 }, { 0x1BAB, 0x1BAD }, { 0x1BE6, 0x1BE6 },
    { 0x1BE8, 0x1BE9 }, { 0x1BED, 0x1BED }, { 0x1BEF, 0x1BF1 },
    { 0x1C2C, 0x1C33 }, { 0x1C36, 0x1C37 }, { 0x1CD0, 0x1CD2 },
    { 0x1CD4, 0x1CE0 }, { 0x1CE2, 0x1CE8 }, { 0x1CED, 0x1CED },
    { 0x1CF4, 0x1CF4 }, { 0x1CF8, 0x1CF9 }, { 0x1DC0, 0x1DF9 },
    { 0x1DFB, 0x1DFF }, { 0x200B, 0x200F }, { 0x202A, 0x202E },
    { 0x2060, 0x2064 }, { 0x2066, 0x206F }, { 0x20D0, 0x20F0 },
    { 0x2CEF, 0x2CF1 }, { 0x2D7F, 0x2D7F }, { 0x2DE0, 0x2DFF },
    { 0x302A, 0x302D }, { 0x3099, 0x309A }, { 0xA66F, 0xA672 },
    { 0xA674, 0xA67D }, { 0xA69E, 0xA69F }, { 0xA6F0, 0xA6F1 },
    { 0xA802, 0xA802 }, { 0xA806, 0xA806 }, { 0xA80B, 0xA80B },
    { 0xA825, 0xA826 }, { 0xA82C, 0xA82C }, { 0xA8C4, 0xA8C5 },
    { 0xA8E0, 0xA8F1 }, { 0xA8FF, 0xA8FF }, { 0xA926, 0xA92D },
    { 0xA947, 0xA951 }, { 0xA980, 0xA982 }, { 0xA9B3, 0xA9B3 },
    { 0xA9B6, 0xA9B9 }, { 0xA9BC, 0xA9BD }, { 0xA9E5, 0xA9E5 },
    { 0xAA29, 0xAA2E }, { 0xAA31, 0xAA32 }, { 0xAA35, 0xAA36 },
    { 0xAA43, 0xAA43 }, { 0xAA4C, 0xAA4C }, { 0xAA7C, 0xAA7C },
    { 0xAAB0, 0xAAB0 }, { 0xAAB2, 0xAAB4 }, { 0xAAB7, 0xAAB8 },
    { 0xAABE, 0xAABF }, { 0xAAC1, 0xAAC1 }, { 0xAAEC, 0xAAED },
    { 0xAAF6, 0xAAF6 }, { 0xABE5, 0xABE5 }, { 0xABE8, 0xABE8 },
    { 0xABED, 0xABED }, { 0xFB1E, 0xFB1E }, { 0xFE00, 0xFE0F },
    { 0xFE20, 0xFE2F }, { 0xFEFF, 0xFEFF }, { 0xFFF9, 0xFFFB },
    { 0x101FD, 0x101FD }, { 0x102E0, 0x102E0 }, { 0x10376, 0x1037A },
    { 0x10A01, 0x10A03 }, { 0x10A05, 0x10A06 }, { 0x10A0C, 0x10A0F },
    { 0x10A38, 0x10A3A }, { 0x10A3F, 0x10A3F }, { 0x10AE5, 0x10AE6 },
    { 0x10D24, 0x10D27 }, { 0x10EAB, 0x10EAC }, { 0x10F46, 0x10F50 },
    { 0x11001, 0x11001 }, { 0x11038, 0x11046 }, { 0x1107F, 0x11081 },
    { 0x110B3, 0x110B6 }, { 0x110B9, 0x110BA }, { 0x110BD, 0x110BD },
    { 0x110CD, 0x110CD }, { 0x11100, 0x11102 }, { 0x11127, 0x1112B },
    { 0x1112D, 0x11134 }, { 0x11173, 0x11173 }, { 0x11180, 0x11181 },
    { 0x111B6, 0x111BE }, { 0x111C9, 0x111CC }, { 0x111CF, 0x111CF },
    { 0x1122F, 0x11231 }, { 0x11234, 0x11234 }, { 0x11236, 0x11237 },
    { 0x1123E, 0x1123E }, { 0x112DF, 0x112DF }, { 0x112E3, 0x112EA },
    { 0x11300, 0x11301 }, { 0x1133B, 0x1133C }, { 0x11340, 0x11340 },
    { 0x11366, 0x1136C }, { 0x11370, 0x11374 }, { 0x11438, 0x1143F },
    { 0x11442, 0x11444 }, { 0x11446, 0x11446 }, { 0x1145E, 0x1145E },
    { 0x114B3, 0x114B8 }, { 0x114BA, 0x114BA }, { 0x114BF, 0x114C0 },
    { 0x114C2, 0x114C3 }, { 0x115B2, 0x115B5 }, { 0x115BC, 0x115BD },
    { 0x115BF, 0x115C0 }, { 0x115DC, 0x115DD }, { 0x11633, 0x1163A },
    { 0x1163D, 0x1163D }, { 0x1163F, 0x11640 }, { 0x116AB, 0x116AB },
    { 0x116AD, 0x116AD }, { 0x116B0, 0x116B5 }, { 0x116B7, 0x116B7 },
    { 0x1171D, 0x1171F }, { 0x11722, 0x11725 }, { 0x11727, 0x1172B },
    { 0x1182F, 0x11837 }, { 0x11839, 0x1183A }, { 0x1193B, 0x1193C },
    { 0x1193E, 0x1193E }, { 0x11943, 0x11943 }, { 0x119D4, 0x119D7 },
    { 0x119DA, 0x119DB }, { 0x119E0, 0x119E0 }, { 0x11A01, 0x11A0A },
    { 0x11A33, 0x11A38 }, { 0x11A3B, 0x11A3E }, { 0x11A47, 0x11A47 },
    { 0x11A51, 0x11A56 }, { 0x11A59, 0x11A5B }, { 0x11A8A, 0x11A96 },
    { 0x11A98, 0x11A99 }, { 0x11C30, 0x11C36 }, { 0x11C38, 0x11C3D },
    { 0x11C3F, 0x11C3F }, { 0x11C92, 0x11CA7 }, { 0x11CAA, 0x11CB0 },
    { 0x11CB2, 0x11CB3 }, { 0x11CB5, 0x11CB6 }, { 0x11D31, 0x11D36 },
    { 0x11D3A, 0x11D3A }, { 0x11D3C, 0x11D3D }, { 0x11D3F, 0x11D45 },
    { 0x11D47, 0x11D47 }, { 0x11D90, 0x11D91 }, { 0x11D95, 0x11D95 },
    { 0x11D97, 0x11D97 }, { 0x11EF3, 0x11EF4 }, { 0x13430, 0x13438 },
    { 0x16AF0, 0x16AF4 }, { 0x16B30, 0x16B36 }, { 0x16F4F, 0x16F4F },
    { 0x16F8F, 0x16F92 }, { 0x16FE4, 0x16FE4 }, { 0x1BC9D, 0x1BC9E },
    { 0x1BCA0, 0x1BCA3 }, { 0x1D167, 0x1D169 }, { 0x1D173, 0x1D182 },
    { 0x1D185, 0x1D18B }, { 0x1D1AA, 0x1D1AD }, { 0x1D242, 0x1D244 },
    { 0x1DA00, 0x1DA36 }, { 0x1DA3B, 0x1DA6C }, { 0x1DA75, 0x1DA75 },
    { 0x1DA84, 0x1DA84 }, { 0x1DA9B, 0x1DA9F }, { 0x1DAA1, 0x1DAAF },
    { 0x1E000, 0x1E006 }, { 0x1E008, 0x1E018 }, { 0x1E01B, 0x1E021 },
    { 0x1E023, 0x1E024 }, { 0x1E026, 0x1E02A }, { 0x1E130, 0x1E136 },
    { 0x1E2EC, 0x1E2EF }, { 0x1E8D0, 0x1E8D6 }, { 0x1E944, 0x1E94A },
    { 0xE0001, 0xE0001 }, { 0xE0020, 0xE007F }, { 0xE0100, 0xE01EF }
};

static const int 
zero_width_count = (sizeof(zero_width)/sizeof(struct interval));
